import socket
import subprocess
import select
import time
import os
import errno
import sys
import platform
import threading
from random import randint as low_quality_randint
mswindows = platform.system()=='Windows'

VERBOSITY = 0
SAVE_UNDECRYPTABLE = False

LOG_ALL_TO_FILE = False

GPG_AVAILABLE = None
GPG_LOCATION = '' # Default is to search the path.
PATH_EXTRA = [] # Extra directories to search for the GPG binary.

def log(*l):
    global LOG_LAST
    LOG_LAST = tuple(l)
    if VERBOSITY:
        print l
    if not LOG_ALL_TO_FILE:
        return
    LOG_ALL_TO_FILE.writelines(l)

def debug(*l):
    if not l:
        print ""
        return
    for i in l[:-1]:
        print i,
    print l[-1]

def slurp_fd_until_eof(fd, buffer):
    s = os.read(fd, 32768)
    l = []
    while s:
        l.append(s)
        s = os.read(fd, 32768)
    buffer.append(''.join(l))


class Slurplines(object):
    def __init__(self,fd):
        self.fd = fd
        self.lines = None
        self.buf = ''
    def __iter__(self):
        while 1:
            if self.lines:
                ret = self.lines[0]
                del self.lines[0]
                yield ret
                ret = None
            elif '\n' in self.buf:
                if self.buf.endswith('\n'):
                    self.lines=self.buf.split('\n')[:-1]
                    self.buf = ''
                else:
                    self.lines = self.buf.split('\n')
                    self.buf = self.lines[-1]
                    del self.lines[-1]
            elif self.fd is not None:
                block = os.read(self.fd,10240)
                if not block:
                    self.fd = None
                    if not self.buf:
                        raise StopIteration
                self.buf += block
                block = None
            elif self.buf:
                yield self.buf

class GPGErrorReason(object):
    # The reason an error occurred.  At the moment, this is very stubby.

    # If expired is not None, it will be the timestamp on which the key
    # expired.
    expired = None
    invalid_recipient = None
    popen_failed = None

    def __repr__(self):
        notes = []
        if self.popen_failed is not None:
            notes.append('popen failed: '+str(self.popen_failed))
        if self.expired is not None:
            notes.append('expired: '+str(self.expired))
        if self.invalid_recipient:
            notes.append('invalid recipient: '+str(self.invalid_recipient))
        if notes:
            return "<GPGErrorReason "+('; '.join(notes))+'>'
        return "<GPGErrorReason: No reason>"

    def IsExpired(self):
        return self.expired is not None

if mswindows: # Seems to also apply on x86-64 laptop
    import _subprocess
    import msvcrt
    noconsole = dict(startupinfo=subprocess.STARTUPINFO())
    noconsole['startupinfo'].dwFlags |= subprocess.STARTF_USESHOWWINDOW
    
    class Pipe(object):
        def __init__(self):
            # Thanks to Digital Engine for the code.
            self.r, self.w = os.pipe()
            curproc = _subprocess.GetCurrentProcess()
            whandle = msvcrt.get_osfhandle(self.w)
            self.whandle = _subprocess.DuplicateHandle(curproc, whandle, curproc, 0, 1, _subprocess.DUPLICATE_SAME_ACCESS)
            rhandle = msvcrt.get_osfhandle(self.r)
            self.rhandle = _subprocess.DuplicateHandle(curproc, rhandle, curproc, 0, 1, _subprocess.DUPLICATE_SAME_ACCESS)
        def wstr(self):
            return str(int(self.whandle))
        def rstr(self):
            return str(int(self.rhandle))
        def CloseW(self):
            os.close(self.w)
            self.whandle.Close()
            self.whandle = None
            self.w = None
        def Close(self):
            if self.w is not None:
                self.CloseW()
            if self.r is not None:
                self.CloseR()
        def CloseR(self):
            os.close(self.r)
            self.rhandle.Close()
            self.r = None
            self.rhandle = None
        def __del__(self):
            # I don't know whether this is a good idea, but I think it is.
            self.Close()
else:
    noconsole = dict()
    class Pipe(object):
        def __init__(self):
            self.r, self.w = os.pipe()
        def wstr(self):
            return str(self.w)
        def rstr(self):
            return str(self.r)
        def CloseW(self):
            os.close(self.w)
            self.w = None
        def CloseR(self):
            os.close(self.r)
            self.r = None
        def Close(self):
            if self.w is not None: self.CloseW()
            if self.r is not None: self.CloseR()
        def __del__(self):
            # I don't know whether this is a good idea, but I think it is.
            self.Close()

class GPGKeyInfo(object):
    long_keyid = None
    handle = None
    key_role = None

    def __repr__(key):
        return '<GPGKeyInfo: '+str(key.long_keyid)+' '+str(key.handle)+' '+str(key.key_role)+'>'

class GPGManager(object):
    require_sig = None
    extra_secring = None
    # Name of binary, FindExecutable() will set the actual correct one.
    executable = 'gpg'
    last_returncode = None
    def __init__(gpg,path,executable=None):
        gpg.path = path
        if executable:
            gpg.executable = executable
        if GPG_LOCATION:
            gpg.executable = os.path.join(GPG_LOCATION,gpg.executable)

    def FindExecutable(gpg,path):
        # Look for a gpg that we can use and set GPGManager.executable
        # accordingly.
        for exe in ('gpg',
                'gpg2',
                os.path.join(path,'gpg'),
                os.path.join(path,'gpg2'),
                os.path.join(path,'gpg.exe'),
                os.path.join(path,'gpg2.exe')):
            if gpg.TryGPGVersion(exe):
                GPGManager.executable = exe
                return True
        print "Error: Could not find GPG."
        return False

    def TryGPGVersion(self,exe):
        if not exe:
            raise Exception("Missing executable.")
        try:
            log("TryGPGVersion: Popen",exe,' --version\n')
            p = subprocess.Popen([exe,'--version'],
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE, **noconsole)
            p.stdin.close()
            p.stdout.read()
            p.stderr.read()
            returncode = p.wait()
            log("TryGPGVersion: return code ",str(returncode),"\n")
            return returncode==0
        except Exception,e:
            print "Exception",e
            try:
                print "Waiting"
                p.wait()
            except Exception,e:
                print "Wait err"
                pass
            return False
        
    def PurgeFiles(gpg):
        # Use this to delete all the GPG datafiles left in the path, except
        # for the random seed.
        try:
            os.unlink(os.path.join(gpg.path,'pubring.gpg'))
        except:
            pass
        try:
            os.unlink(os.path.join(gpg.path,'secring.gpg'))
        except:
            pass
        try:
            os.unlink(os.path.join(gpg.path,'pubring.gpg~'))
        except:
            pass
        try:
            os.unlink(os.path.join(gpg.path,'trustdb.gpg'))
        except:
            pass

    def EncryptAndSign(gpg,recipient,message,identity):
        args = [gpg.executable,'--homedir='+gpg.path,
            '--secret-keyring='+gpg.signing_sec_keyring,
            '--keyring='+gpg.signing_pub_keyring,
            '--trust-model','always',
            '--throw-keyids',
            # For bandwidth, not privacy.  The performance hit is marginal
            # in the tunnel case
            # We're assuming that the pre-shared keys are trustworthy.
            '--sign','--local-user',identity,
            '--encrypt','--recipient',recipient]
        log("EncryptAndSign: ",str(args),"\n")
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        p.stdin.write(message)
        p.stdin.close()
        data = p.stdout.read()
        errs = p.stderr.read()
        returncode = p.wait()
        log("return code: "+str(returncode)+"\n")
        return data
        
    def Encrypt(gpg,recipient,message,throw_keyids=True):
        # Switching statr and statw did not make it work in real Windows.
        statpipe = Pipe()
        gpg.last_error = GPGErrorReason()
        args = [gpg.executable,'--homedir='+gpg.path,
            '--status-fd='+statpipe.wstr(),
            '--trust-model','always',
            ]
        if throw_keyids:
            args.append('--throw-keyids')
            # For bandwidth, not privacy.  The performance hit is marginal
            # in the tunnel case
            # We're assuming that the pre-shared keys are trustworthy.
        args.extend((
            '--encrypt','--recipient',recipient))
        if VERBOSITY:
            print "Running",args
        log("Encrypt: ",str(args),"\n")
        try:
            p = subprocess.Popen(args,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE, **noconsole)
        except Exception,e:
            gpg.last_error.popen_failed_args = tuple(args)
            gpg.last_error.popen_failed = e
            return
        statpipe.CloseW()

        # We wrap this around .communicate() to prevent deadlocks when the
        # pipes are full.
        statpipe_data = []
        statpipe_thread = threading.Thread(
            target=slurp_fd_until_eof,
            args=(statpipe.r, statpipe_data))
        statpipe_thread.setDaemon(True)
        statpipe_thread.start()
        data, errs = p.communicate(input=message)
        statpipe_thread.join()
        if statpipe_data:
            status = statpipe_data[0]
        else:
            print "Error: No statpipe data?"
            print statpipe.r.read()
        statpipe.Close()

        for line in status.split("\012"):
            if VERBOSITY>1:
                print line
            log(line+"\012")
            if   line.startswith('[GNUPG:] KEYEXPIRED '):
                gpg.last_error.expired = int(line[20:])
            elif line.startswith('[GNUPG:] INV_RECP '):
                if VERBOSITY:
                    print "249:<"+line[18:]+">"
                gpg.last_error.invalid_recipient = line[18:]
        gpg.last_status = status
        log('stderr = ',errs,"\n")
        gpg.last_stderr = errs
        gpg.last_returncode = p.wait()
        log("returncode = "+str(gpg.last_returncode)+"\n")
        return data

    def DeletePublicKey(gpg, keyid):
        args = [ gpg.executable, '--homedir='+gpg.path,
            '--delete-key', keyid ]
        p = subprocess.Popen(args, **noconsole)
        p.wait()
    def DeleteSecretAndPublicKey(gpg,keyid):
        args = [ gpg.executable, '--homedir='+gpg.path,
            '--delete-secret-and-public-key', keyid ]
        p = subprocess.Popen(args, **noconsole)
        p.wait()

    def Decrypt(gpg,message):
        statpipe = Pipe()
        args = [gpg.executable,'--homedir='+gpg.path,
            '--decrypt',
            '--status-fd='+statpipe.wstr()]
        if gpg.extra_secring:
            args.append('--secret-keyring='+gpg.extra_secring)
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        statpipe.CloseW()
        p.stdin.write(message)
        p.stdin.close()
        notdone = True
        data = p.stdout.read()
        status = os.read(statpipe.r,40960)
        statpipe.Close()
        if gpg.require_sig:
            for line in status.split('\012'):
                if line.startswith('[GNUPG:] VALIDSIG '):
                    if line.split(' ',3)[2]==gpg.require_sig:
                        p.wait()
                        return data
            if VERBOSITY:
                print "DEBUG: no valid signature on data"
            p.wait()
            return None
        p.wait()
        gpg.last_status = status
        return data

    def Decrypt2(gpg,message):
        statpipe = Pipe()
        args = [gpg.executable,'--homedir='+gpg.path,
            '--decrypt',
            '--status-fd='+statpipe.wstr()]
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        statpipe.CloseW()
        p.stdin.write(message)
        p.stdin.close()
        notdone = True
        data = p.stdout.read()
        enc_to = None
        decryption_okay = False
        sig_validated = False
        gpg.last_status = []
        for line in Slurplines(statpipe.r):
            if line.startswith('[GNUPG:] ENC_TO '):
                enc_to = line.split(' ')[2]
            elif line=='[GNUPG:] DECRYPTION_OKAY' and enc_to:
                decryption_okay = True
            elif line.startswith('[GNUPG:] VALIDSIG '):
                if line.split(' ',3)[2]==gpg.require_sig:
                    sig_validated = True
            gpg.last_status.append(line)
        statpipe.Close()
        if not decryption_okay:
            enc_to = None
        if gpg.require_sig:
            if not sig_validated:
                print "Error: Signature not validated."
                p.wait()
                return None,enc_to
        p.wait()
        return data,enc_to

    def VerifyEncrypted(gpg,keyid,message,keyring):
        statpipe = Pipe()
        args = [gpg.executable,'--homedir='+gpg.path+'/',
            '--secret-keyring='+keyring,
            '--decrypt','--trust-model','always',
            '--status-fd='+statpipe.wstr()
            ]
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        statpipe.CloseW()
        p.stdin.write(message)
        p.stdin.close()
        line = os.read(statpipe.r,10240)
        validated = False
        notdone = True
        lines = ()
        while notdone:
            if lines:
                line = lines[0]
                del lines[0]
            elif line.endswith("\n"):
                line = line[:-1]
            if "\n" in line:
                lines = line.split("\n")
            if line.startswith('[GNUPG:] VALIDSIG '):
                splitline = line.split(' ')
                if splitline[2]==keyid:
                    validated = True
                    notdone = False
            if not lines:
                line = os.read(statpipe.r,10240)
                if not line:
                    notdone = False
        statpipe.Close()
        p.wait()
        return validated

    def Verify(gpg,keyid,message):
        statpipe = Pipe()
        print "Checking signature of packet in",gpg.path
        args = [gpg.executable,'--homedir='+gpg.path+'/',
            '--verify','--trust-model','always',
            '--status-fd='+statpipe.wstr()
            ]
        print "args=",args
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        statpipe.CloseW()
        p.stdin.write(message)
        p.stdin.close()
        line = os.read(statpipe.r,10240)
        validated = False
        notdone = True
        lines = ()
        while notdone:
            if lines:
                line = lines[0]
                del lines[0]
            elif line.endswith("\n"):
                line = line[:-1]
            if "\n" in line:
                lines = line.split("\n")
            if line.startswith('[GNUPG:] VALIDSIG '):
                splitline = line.split(' ')
                if splitline[2]==keyid:
                    validated = True
                    notdone = False
            if not lines:
                line = os.read(statpipe.r,10240)
                if not line:
                    notdone = False
        statpipe.Close()
        p.wait()
        return validated

    def Import(gpg,block):
        """
        Import a key from the data block to the keyring; return a key info
        object, or None if import failed.  The key info is guaranteed to
        contain a long_keyid member, but not necessarily a handle or key role.
        """
        try:
            os.mkdir(gpg.path)
        except OSError,e:
            if e.errno!=errno.EEXIST:
                raise e
        statpipe = Pipe()
        args = [gpg.executable,'--homedir='+gpg.path,'--import',
            '--status-fd='+statpipe.wstr()
            ]
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        statpipe.CloseW()
        p.stdin.write(block)
        p.stdin.close()
        ret = GPGKeyInfo()
        imported = False
        for line in Slurplines(statpipe.r):
            if line.endswith('\012'): line = line[:-1]
            if line.startswith('[GNUPG:] IMPORT_OK '):
                line = line.split(' ')
                ret.long_keyid = line[3]
                imported = True
            elif line.startswith('[GNUPG:] IMPORTED '):
                line = line.split(' ')
                if len(line)>3:
                    ret.handle = line[3]
                    if ret.handle.endswith('(recv)'):
                        ret.handle = ret.handle[:-6]
                        ret.key_role = '(recv)'
        statpipe.Close()
        p.wait()
        if not imported:
            return None
        return ret

    def Export(gpg,keyid):
        if keyid is None:
            print "ERROR: no keyid specified."
            return ''
        args = [gpg.executable, '--homedir='+gpg.path,
            '--export',keyid ]
        p = subprocess.Popen(args,
            stdin=subprocess.PIPE, stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        ret = p.stdout.read()
        p.wait()
        return ret

    def Sign(gpg,block,keyid):
        # Return block signed with key keyid, or None if the signing fails.
        args = [gpg.executable,'--homedir='+gpg.path,'--sign',
            '-u',keyid,
            ]
        p = subprocess.Popen(args,
            stdin = subprocess.PIPE,
            stdout = subprocess.PIPE,
            stderr = subprocess.PIPE, **noconsole)
        p.stdin.write(block)
        p.stdin.close()
        ret = p.stdout.read() or None
        p.wait()
        return ret

    def GenerateKeypair(gpg,size=4096,valid=0,name='',primegen_cb=None):
        # Attempt to generate a keypair, return None if it does not exist.
        if len(name)<5:
            return None
        try:
            os.mkdir(gpg.path)
        except OSError,e:
            if e.errno!=errno.EEXIST:
                raise e
        statpipe = Pipe()
        cmdpipe = Pipe()
        args = [gpg.executable,'--homedir='+gpg.path,'--gen-key',
            '--status-fd='+statpipe.wstr(),
            '--command-fd='+cmdpipe.rstr(),
            ]
        p = subprocess.Popen(args, **noconsole)
        statpipe.CloseW()
        cmdpipe.CloseR()
        notdone = True
        ret = None
        lines = []
        while notdone:
            if lines:
                line = lines[0]
                del lines[0]
            else:
                line = os.read(statpipe.r,4096)
            if line.endswith('\n'):
                line = line[:-1]
            if '\n' in line:
                lines = line.split('\n')
                line = lines[0]
                del lines[0]
            if line=='[GNUPG:] GET_HIDDEN passphrase.enter':
                # This is one of the places that passphrase support would show
                # up in if we had it...
                os.write(cmdpipe.w,"\n")
            elif line=='[GNUPG:] GET_LINE keygen.algo':
                os.write(cmdpipe.w,"1\n")
            elif line=='[GNUPG:] GET_LINE keygen.comment':
                os.write(cmdpipe.w,"\n")
            elif line=='[GNUPG:] GET_LINE keygen.email':
                os.write(cmdpipe.w,"\n")
            elif line=='[GNUPG:] GET_LINE keygen.name':
                os.write(cmdpipe.w,name+"\n")
            elif line=='[GNUPG:] GET_LINE keygen.size':
                os.write(cmdpipe.w,str(size)+"\n")
            elif line=='[GNUPG:] GET_LINE keygen.valid':
                os.write(cmdpipe.w,str(valid)+"\n")
            elif line.startswith('[GNUPG:] KEY_CREATED B '):
                ret = line[23:]
                notdone = False
            elif line.startswith('[GNUPG:] PROGRESS primegen '):
                if primegen_cb:
                    primegen_cb(line[27:])
            elif (line not in ('[GNUPG:] GOT_IT', '[GNUPG:] GOOD_PASSPHRASE',
                '[GNUPG:] MISSING_PASSPHRASE')) and (not line.startswith(
                '[GNUPG:] NEED_PASSPHRASE_SYM')):
                print "ERROR: Read unexpected line",line,"from GPG"
        statpipe.Close()
        cmdpipe.Close()
        p.wait()
        return ret

    def DetermineEncTo(gpg, keyid):
        dummymessage = gpg.Encrypt(keyid,"correct horse battery staple.",throw_keyids=False)
        return gpg.DetermineDestination(dummymessage)

    def DetermineDestination(gpg,message):
        statpipe = Pipe()
        # XXX check to make sure it doesn't use any of the default keyring
        # directories.
        p = subprocess.Popen((gpg.executable,'--list-packets',
            '--status-fd='+statpipe.wstr()),
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, **noconsole)
        statpipe.CloseW()
        p.stdin.write(message)
        p.stdin.close()
        lines = ()
        ret = None
        for line in Slurplines(statpipe.r):
            if line.startswith('[GNUPG:] ENC_TO '):
                ret = line.split(' ')[2]
        p.wait()
        statpipe.Close()
        return ret
        #gpg --list-packets --status-fd=n
        #will return the short keyid in an ENC_TO line.  We can/should omit
        #all keyring and trustdb use.
        #[GNUPG:] ENC_TO 709D8CA991467CA5 1 0

def sanitize_bytes(pkt,num=0):
    if num:
        pkt = pkt[:num]
    s = ""
    for c in pkt:
        if ord(c)>=33 and ord(c)<=126 and c!='\\':
            s += c
        else:
            s += "\\x%02x"%ord(c)
    return s

def FindGPG(gpghome,recheck=False):
    global GPG_AVAILABLE
    global GPG_LOCATION
    if VERBOSITY:
        print "FindGPG() started w/GPG_LOCATION=",GPG_LOCATION, 'recheck',recheck
    ctx = GPGManager(gpghome)

    if GPG_AVAILABLE is None or recheck:
        GPG_AVAILABLE = ctx.TryGPGVersion('gpg')
        if VERBOSITY:
            print "gpg.py:651 GPG_AVAILABLE=",GPG_AVAILABLE
        if GPG_AVAILABLE:
            return True
        if (not GPG_AVAILABLE) and PATH_EXTRA:
            for d in PATH_EXTRA:
                if ctx.FindExecutable(d):
                    if VERBOSITY:
                        print "Found executable in",d
                    GPG_LOCATION = d
                    GPG_AVAILABLE = True
                    return True

        GPG_AVAILABLE = ctx.TryGPGVersion('gpg2')
    
        if (not GPG_AVAILABLE) and PATH_EXTRA:
            for d in PATH_EXTRA:
                if ctx.FindExecutable(d):
                    GPG_LOCATION = d
                    GPG_AVAILABLE = True
                    return True
    return GPG_AVAILABLE
