# nopickles.py is copyright 2014 Spiky Caterpillar and released under the
# WTFPL.

# These are the rather hackish routines to replace Ren'Py's use of Python
# pickles.  Note that while this is a lot more secure than pickle, it's not
# currently fully secure.  In particular, note that it ships with
# INSECURE=True set.

# Before you panic, note that the only way I know of to actually exploit
# the pickles is for someone to have write access to your saved games or
# persistent data - so things like trading saves and persistent, or letting
# someone/something else write your files, or using networked providers
# to store your saves.  So it's a real risk, but only a HUGE one if I were
# using some form of networked save system.

# Actually using this requires some minor changes to Ren'Py (diffing the
# Date Warp Ren'Py against a stock 6.17.7 should tell you what these changes
# are) and constraining your code to only the syntax that the nopickles
# inspector handles.  If you're seriously planning on using it, you may want
# to talk it over with Spiky for integration advice.

import base64
import platform
import zlib

import renpy
import unroller
import os
from namedtransform import *
import sanitize
import sanitize_wh
import xlat
#import animation

# Used in nopickles.rpy to determine whether or not to dump jpegs.
SAVE_SCREENSHOTS = True

INSECURE = False
DEBUG = True

# Set to True to include a LOT more debug info in saves.
VERBOSE_SAVES = False

# If set, autosave on every single line for debugging purposes.
AUTOHAMMER = False

FUTUREDEBUG = True # Set to True to provide debug information about things
# which are not a problem for Date Warp but may be a problem for future games.

# Warnings are a list of sequences; the first item of the sequence is the
# severity, the second and subsequent items are data.
#
# Severities:
# 3: Discarded data - game should still play fine, nothing of value was lost.
#    Likely to be an automatic conversion between revertable and Python sets
#    or something similar.
# 5: Discarded data - something missing may break the game.
#    Accepted known-bad, but not security-hole-causing data - the data may
#    break gameplay but won't cause security holes.
# 6: Accepted tainted data.
class Warnings(list):
    def __init__(self):
        list.__init__(self)
        self.notes = []
    def note(self,v):
        self.notes.append(v)
    def unnote(self):
        self.notes.pop()
    def append(self, v):
        list.append(self,v+tuple(self.notes))

def clear_warnings():
    global warnings
    warnings = Warnings()

clear_warnings()

def most_important_warning():
    highest = -1
    for warning in warnings:
        if warning[0]>highest:
            highest = warning[0]
            highest_val = warning
    if highest>=0:
        ret = repr(highest_val)
        return ret.replace('{','{{')
    return '(No warnings.)'

def serialize_dict_contents(item,rollobjmemo=None):
    ret = []
    for k, v in item.iteritems():
        ret.extend(serialize_value(k,rollobjmemo=rollobjmemo))
        ret.extend(serialize_value(v,rollobjmemo=rollobjmemo))
    return ret

class TypeHolder():
    def method(self):
        pass
bound_instancemethod_type = type(TypeHolder().method)

def serialize_callable(cb, memoset, rollobjmemo):
    if type(cb) is bound_instancemethod_type:
        ret = ['(']
        ret.append('str bound '+cb.__name__)
        try:
            ret.extend(serialize_value(cb.im_self, memoset=memoset, rollobjmemo=rollobjmemo))
        except Exception, e:
            serialize_warnings.append("Failed to serialize bound instancemethod "+cb.__name__+repr(cb.im_self)+': '+repr(e))
        ret.append(')')
        return ret
    elif type(cb) is type(serialize_callable):
        return 'str func '+cb.__module__+'.'+cb.__name__ ,
    serialize_warnings.append(("serialize_callable failed on",cb))
    return '# serialize_callable failed on: '+repr(cb), 

def serialize_value(item,memoset=None,rollobjmemo=None):
    ti = type(item)
    if ti is tuple:
        ret = ['(',]
        for i in item:
            ret.extend(serialize_value(i,rollobjmemo=rollobjmemo))
        ret.append(')')
        return ret
    elif item is None:
        return 'None',
    elif ti is str or ti is unicode:
        return 'str '+escape_str(item),
    elif ti is xlat.translated:
        return 'xtr '+escape_str(item),
    elif ti is int or ti is long:
        return 'int '+str(item),
    elif ti is float:
        return 'float '+str(item),
    elif ti is bool:
        return str(item),
    elif ti is dict:
        ret = ['p{',]+serialize_dict_contents(item,rollobjmemo=rollobjmemo)+['}',]
        return ret
    elif ti is set:
        ret = ['s(']
        for member in item:
            ret.extend(serialize_value(member))
        ret.append(')')
        return ret
    elif ti is renpy.python.RevertableDict:
        ret = ['{']
        ret.extend(serialize_dict_contents(item,rollobjmemo=rollobjmemo))
        return ret + ['}',]
    elif ti is renpy.python.RevertableList:
        ret = ['[',]
        for i in item:
            ret.extend(serialize_value(i,memoset=memoset,rollobjmemo=rollobjmemo))
        return ret + [']',]
    elif ti is NamedTransform:
       return ('trans '+item.name,)
    elif isinstance(item,SanitizableRevertableObject):
        try:
            return item.Serialize(memoset,rollobjmemo=rollobjmemo)
        except Exception,e:
            serialize_warnings.append("Exception "+repr(e)+" serializing a "+str(type(item)))
    warning = "BUG: Cannot serialize values of type "+str(type(item))+" "+str(item)
    warning = warning.replace('{','(*')
    serialize_warnings.append(warning)
    print warning
    if AUTOHAMMER:
        print item
        raise Exception('Serialization error.')
    return ('#.44 unhandled value'+str(type(item)),)

def serialize_tnv(k, v, memoset=None, memoid=None, rollobjmemo=None):
    # Serialize a key/value pair to a sequence of one or more strings.
    # The format is type, name, value.
    # Non-scalar values will extend over multiple strings in the returned
    # sequence.
    # Not entirely sure why I was warning about the lack of a rollback object
    # memo here.
    if type(v) is str or type(v) is unicode:
        return 'str '+k+' '+escape_str(v),
    elif type(v) is int:
        return 'int '+k+' '+str(v),
    elif type(v) is bool:
        return str(v)+' '+k,
    elif type(v) is float:
        return 'float '+k+' '+str(v),
    elif type(v) is tuple:
        ret = ['( '+k]
        for i in v:
            ret.extend(serialize_value(i, memoset=memoset,rollobjmemo=rollobjmemo))
        ret.append(')')
        return ret
    elif v is None:
        return 'None '+k,
    elif type(v) is renpy.python.RevertableDict:
        ret = ['{ '+k]
        for dk, dv in v.iteritems():
            ret.extend(serialize_value(dk,memoset=memoset, rollobjmemo=rollobjmemo))
            ret.extend(serialize_value(dv,memoset=memoset,rollobjmemo=rollobjmemo))
        ret.append('}')
        return ret
    elif type(v) is dict:
        ret = ['p{ '+k]
        for dk, dv in v.iteritems():
            ret.extend(serialize_value(dk,rollobjmemo=rollobjmemo))
            ret.extend(serialize_value(dv,memoset=memoset,rollobjmemo=rollobjmemo))
        ret.append('}')
        return ret
    elif type(v) is renpy.python.RevertableList:
        ret = ['[ '+k]
        if rollobjmemo is None:
            ret.append('#.160: obmemo is None')
        else:
            rolledv = renpy.python.RevertableList()
            if id(v) in rollobjmemo:
                print "WARNING 198", id(v),
                rolledv._rollback(rollobjmemo[id(v)])
                ret.append("#.179"+str(rolledv))
            if VERBOSE_SAVES:
                ret.append("#.164 "+str(rolledv))
        for i in v:
            ret.extend(serialize_value(i,memoset=memoset,rollobjmemo=rollobjmemo))
        ret.append(']')
        return ret
    elif type(v) is renpy.python.RevertableSet:
        ret = ['rs( '+k]
        for i in v:
            ret.extend(serialize_value(i,memoset=memoset, rollobjmemo=rollobjmemo))
        ret.append(')')
        return ret
    elif type(v) is set:
        ret = ['s( '+k]
        for i in v:
            ret.extend(serialize_value(i,rollobjmemo=rollobjmemo))
        ret.append(')')
        return ret
    elif type(v) is renpy.python.StoreDeleted:
        return ('del '+k,)
    elif isinstance(v, SanitizableRevertableObject):
        return v.Serialize(memoset,infix=' '+k,memoid=memoid, rollobjmemo=rollobjmemo)
    elif isinstance(v, NamedTransform):
        return 'xform '+k+' '+escape_str(v.name),
    else:
        return '# Invalid tnv: '+k+':'+str(type(v)),
        
def serialize_script_loc(loc):
    # We reverse the order of locations so they're line, timestamp, file
    # so that files with spaces will not screw up parsing.
    if type(loc) is not tuple:
        if type(loc) is str:
            return 'lbl '+escape_str(loc)
        serialize_warnings.append("Not sure how to serialize location "+repr(loc))
    if len(loc)==4:
        if loc[3] not in ALLOWED_LOCATION_4TUPLE_SUFFICES:
            serialize_warnings.append("Not sure how to serialize location "+repr(loc))
        return str(loc[3])+' '+str(loc[2])+' '+str(loc[1])+' '+str(loc[0])
    if len(loc)!=3:
        serialize_warnings.append("Script location "+repr(loc)+" length unexpected.")
    return str(loc[2])+' '+str(loc[1])+' '+str(loc[0])

def serialize_rollback_log(log):
    roots = renpy.game.log.freeze()
    #roots = log.get_roots()
    ret = ['log',]
    found = False
    l = list(log.log)
    l.reverse()
    elongatable = []
    for ent in l:
        #ret.append("#.156: "+str(roots))
        es = SerializedRollbackLogEntry(ent, roots)
        if es.can_be_last():
            roots = False
            found = True
        elongatable.append(es)
    elongatable.reverse()
    for es in elongatable:
        ret.extend(es.elongate())
    ret.append('# end rollback log')
    ret.append('end')
    if not found:
        print log.forward_info(),247,renpy.exports.roll_forward_info()
        try:
            f = file(os.path.join(renpy.config.savedir, 'noserialization.whs'),'wb')
            f.writelines('\n'.join(ret))
            f.close()
            unroller.traceback_include_save = 'noserialization.whs'
        except Exception, e:
            print "Error saving save-failure data:",e
        unroller.noserialization_err = tuple([(SerializedRollbackLogEntry(ent, roots), ent) for ent in l])
        raise Exception("No valid serialization points found.")
    return ret

def serialize_context_musicchannel(channelname, data):
    ret = []
    if data.last_filenames:
        for fn in data.last_filenames:
            ret.append('file '+escape_str(fn))
        if data.secondary_volume!=1.0:
            ret.append('vol '+str(data.secondary_volume))
        if data.pan!=0:
            print "Warning: nopickles does not currently save pan values."
    if ret:
        return ['channel '+str(channelname),]+ret+['end',]
    return ()

def serialize_context_music(music):
    # A context's music is a Ren'Py RevertableDict.  I'm not entirely sure why.
    ret = []
    for k,v in music.iteritems():
        ret.extend(serialize_context_musicchannel(k,v))
    if ret:
        return ['music',]+ret+['end',]
    return ()

ROLLBACK_OBJECT_TYPECODES={
    renpy.python.RevertableObject:'ro'
    }

def serialize_rollback_object(obj, data):
    to = type(obj)
    if to not in ROLLBACK_OBJECT_TYPECODES:
        return '#.155'+str(type(obj)),
    ret = ["%x"%id(obj)+" "+ROLLBACK_OBJECT_TYPECODES[to],]
    return ret+serialize_value(data)

ALLOWED_LOCATION_4TUPLE_SUFFICES = ('translate', 'end_translate')

class SerializedRollbackLogEntry(object):
    location_makes_sense = False
    has_roots = False
    def __init__(self, entry, roots):
        self.renpy_entry = entry
        cloc = entry.context.current
        # We used to return unless entry.checkpoint was true, in an effort to
        # save space by avoiding saving non-checkpoint items.  However, this
        # caused the current line's log entry not to get saved, which would in
        # turn mean that the game would always load with the variables set as
        # they were when the game was saved, but the location set to the prior
        # one - which, in turn, led to re-execution of any code that happened
        # between the previous line of dialog and the one on which the game
        # was saved.

        # We don't save rollback log entries if the location doesn't make sense.
        if type(cloc) is not tuple:
            return
        if len(cloc) < 3 or len(cloc)>4:
            return
        if len(cloc)!=3 and cloc[-1] not in ALLOWED_LOCATION_4TUPLE_SUFFICES:
            raise Exception("Unexpected rollback log entry tail "+cloc)
            return
        if type(cloc[2]) is not int or type(cloc[1]) is not int or type(cloc[0]) not in (str, unicode):
            return
        self.location_makes_sense = True
        self.data = ret = ['entry','callstack']
        memoset = set()
        i = 0
        if DEBUG:
            if len(entry.context.call_location_stack)!=len(entry.context.return_stack):
                raise Exception("Error: call stack size mismatch.")
        while i<len(entry.context.call_location_stack):
            ret.append(serialize_script_loc(entry.context.call_location_stack[i]))
            rse = entry.context.return_stack[i]
            if type(rse) is tuple:
                ret.append('ret '+serialize_script_loc(rse))
            else:
                ret.append('retlbl '+escape_str(rse))
            i += 1
        ret.append('end')
        ret.extend(serialize_context_music(entry.context.music))
        objmemo = {}
        for k,v in entry.objects:
            objmemo[id(k)] = v
        for item in entry.context.dynamic_stack:
            # These are regular dicts.
            ret.append('dynamic')
            for k,v in item.iteritems():
                if k not in SERIALIZE_STORE_OVERRIDES:
                    ret.extend(serialize_tnv(k,v,memoset=memoset,rollobjmemo=objmemo))
                else:
                    ret.append('# Debug: 303')
                    ret.extend(SERIALIZE_STORE_OVERRIDES[k])
            ret.append('end')
        ret.append('current '+serialize_script_loc(cloc))
        ret.append('#168.')
        for storename, storeval in entry.stores.iteritems():
            if storeval:
                ret.append('store '+storename)
                for k,v in storeval.iteritems():
                    if k not in SERIALIZE_STORE_OVERRIDES:
                        ret.extend(serialize_tnv(k,v,memoset=memoset,rollobjmemo=objmemo))
                    else:
                        ret.extend(SERIALIZE_STORE_OVERRIDES[k](k,v,memoset, rollobjmemo=objmemo))
                ret.append('# end stores')
                ret.append('end')
        # Behind experimental rollback-related saving.
        #if entry.objects:
        #    ret.append('objects')
        #    for obj, data in entry.objects:
        #       ret.extend(serialize_rollback_object(obj, data))
        #    ret.append('end')
        # End experimental rollback-related saving.
        self.root_chunk = ['#.root chunk']
        try:
            if roots:
                for k,v in roots.iteritems():
                    #try:
                    #    print "Debug:",k,v
                    #except Exception,e:
                    #    print "Exception printing debug message:",e
                    if id(v) in objmemo:
                        # Commented this out because it crashes if loading a save containing newlines
                        # in the string.
                        #self.root_chunk.append('#.211 rollback '+str(v)+'->'+str(objmemo[id(v)]))
                        if isinstance(v, SanitizableRevertableObject):
                            nv = v.RolledBackClone(objmemo[id(v)])
                        else:
                            try:
                                nv = type(v)()
                                nv._rollback(objmemo[id(v)])
                                self.root_chunk.append('#:332'+str(nv))
                                self.root_chunk.append('#:333'+str(v))
                            except Exception,e:
                                nv = None
                                raise Exception("An exception occurred serializing "+repr(k)+' '+str(type(v))+" "+repr(v)+": "+str(e))
                        ret.append('#memo')
                        if k in STORE_SERIALIZERS:
                            ret.extend(STORE_SERIALIZERS[k](k, nv, memoset, rollobjmemo=objmemo))
                        else:
                            self.root_chunk.extend(serialize_tnv(k,nv,memoset=memoset,memoid=id(v),rollobjmemo=objmemo))
                    elif k in SERIALIZE_STORE_OVERRIDES:
                        self.root_chunk.extend(SERIALIZE_STORE_OVERRIDES[k])
                    elif k in STORE_SERIALIZERS:
                        ret.append('# custom serializer for '+k+' '+repr(v))
                        ret.extend(STORE_SERIALIZERS[k](k, v, memoset, rollobjmemo=objmemo))
                    else:
                        self.root_chunk.extend(serialize_tnv(k,v,memoset=memoset,rollobjmemo=objmemo))
            self.has_roots = True
            self.root_chunk.append('#/root chunk')
        except Exception,e:
            ret.append("# exception "+repr(e)+" "+type(e)+" serializing.")
            serialize_warnings.append(ret[-1])
            if DEBUG:
                print "Debug: Could not serialize properly.",e
                raise
            self.root_exception = e
        ret.extend(serialize_scene_lists(entry.context.scene_lists))
    def can_be_last(self):
        return self.location_makes_sense and self.has_roots

    # removed objects_chunk because it's rendered obsolete by the fact that
    # we actually roll back things in the saves.

    def elongate(self):
        if not self.location_makes_sense:
            return ()
        if self.has_roots:
            return self.data + self.root_chunk + ['end',]
        return self.data+['end',]

alphalow = 'abcdefghijklmnopqrstuvwxyz'

def clean_string(s, allow, replace='?'):
    ret = ''
    for c in s:
        if c in allow:
            ret += c
        else:
            ret += replace
    return ret

def unserialize_value(context_stack, line):
    if line=='(':
        context_stack.append(SafeloadTupleValueContext())
        return context_stack[-1]
    elif line=='[':
        context_stack.append(SafeloadRevertableListValueContext())
        return context_stack[-1]
    elif line=='p{':
        context_stack.append(SafeloadPythonDictValueContext())
        return context_stack[-1]
    elif line=='{':
        context_stack.append(SafeloadRevertableDictValueContext())
        return context_stack[-1]
    elif line=='s(':
        context_stack.append(SafeloadPythonSetValueContext())
        return context_stack[-1]
    elif line=='False':
        return False
    elif line=='None':
        return None
    elif line=='True':
        return True
    elif line.startswith('float '):
        return float(line[6:])
    elif line.startswith('int '):
        return int(line[4:])
    elif line.startswith('sro '):
        context_stack.append(SafeloadSanitizableRevertableObjectContext(line[4:], lineno=line.num))
        return context_stack[-1]
    elif line.startswith('sroref '):
        memoctx = get_stack_memo_context(context_stack)
        return memoctx.OBJECT_MEMO[i64dec(line[7:])]
    elif line.startswith('str '):
        return unescape_str(line[4:])
    elif line.startswith('trans '):
        trans = getattr(renpy.store,line[6:],None)
        if type(trans) is NamedTransform:
            return trans
        else:
            warnings.append((5,"unserialize_value discarded missing transform",trans))
            ret = getattr(renpy.store,'unresolvable')
            if type(ret) is not NamedTransform:
                raise Exception("renpy.store.default is not a NamedTransform.")
            return ret
    elif line.startswith('xtr '):
        return xlat.translated(unescape_str(line[4:]))
    elif line.startswith('ac '):
        candidate = getattr(animation,line[3:])
        if type(candidate) is type(animation.AnimController) and animation.AnimController in candidate.mro():
            return candidate()
        warnings.append((5,"unserialize_value used NullAnimController for",line))
        return animation.NullAnimController()
        
    raise Exception("Debug: unserialize_value discarded '"+line+"'")
    return None

class DeindirectableContext(object):
    """
    Used to check to see whether something should be deindirected before use.

    Note that deindirection is meant to be recursive; a deindirected value should
    contain no deindirectable values.
    """
    pass

class SafeloadSequenceValueContext(DeindirectableContext):
    """
    This isn't meant to be used directly; rather, it's a superclass for
    tuples and lists.
    """
    lineno = 0
    last_lineno = 0
    def __repr__(self):
        return '<'+type(self).__name__+(' %x %d:%d'%(id(self),self.lineno,self.last_lineno))+'>'

    def __init__(self):
        self.l = []

    def deindirect_values(self):
        i = 0
        while i<len(self.l):
            if isinstance(self.l[i], SafeloadSequenceValueContext):
                self.l[i] = self.l[i].deindirect()
            if type(self.l[i]) is SafeloadPythonDictValueContext:
                self.l[i] = self.l[i].deindirect()
            i += 1

class SafeloadRevertableListValueContext(SafeloadSequenceValueContext):
    def handle(self,context_stack,line):
        if not self.lineno:
            try:
                self.lineno = line.num
            except Exception, e:
                print "No line number for", line, type(line)
                raise e
        if line==']':
            self.last_lineno = line.num
            context_stack.pop()
            # We used to deindirect list values when done, but that had the
            # unpleasant side effect of ignoring data serialized after a
            # Level's grid - probably caused by the recursive srorefs?
            # self.deindirect_values()
            return
        self.l.append(unserialize_value(context_stack, line))
    def deindirect(self):
        return renpy.python.RevertableList([deindirect_if_needed(i) for i in self.l])

class SafeloadPythonDictValueContext(DeindirectableContext):
    def __init__(self):
        self.l = []
    def handle(self, context_stack, line):
        if line=='}':
            context_stack.pop()
            return
        self.l.append(unserialize_value(context_stack, line))
    def build_dict(self):
        i = 0
        d = {}
        while i<len(self.l):
            d[deindirect_if_needed(self.l[i])] = deindirect_if_needed(self.l[i+1])
            i += 2
        return d
    def deindirect(self):
        return self.build_dict()

class SafeloadRevertableDictValueContext(SafeloadPythonDictValueContext):
    def build_dict(self):
        raise Exception("Wrong type.")
    def build_revertable_dict(self):
        i = 0
        d = renpy.python.RevertableDict()
        while i+1<len(self.l):
            d[deindirect_if_needed(self.l[i])] = deindirect_if_needed(self.l[i+1])
            i += 2
        if i<len(self.l):
            warnings.append((5,'Dict key without value discarded',self.l[i]))
        return d
    def deindirect(self):
        return self.build_revertable_dict()

def get_stack_memo_context(stack):
    ret = None
    for ent in stack:
        if type(ent) is SafeloadRollbackLogEntryContext:
            if ret is None:
                ret = ent
            else:
                warnings.append((4,'RollbackLogEntry inside RollbackLogEntry, using outermost for memo context.'))
    return ret

SRO_MODULES = renpy.store,

class SafeloadSanitizableRevertableObjectContext(DeindirectableContext):
    actual_object = None
    sanitizable = True

    last_lineno = 0

    def __init__(self, typename, lineno=0):
        self.lineno = lineno
        if typename[0] not in OK_VARNAME_FIRST:
            warnings.append((4,'typename',typename,'c',c, 'not in',OK_VARNAME_FIRST))
            self.sanitizable = False
            t = renpy.python.deleted
        else:
            for c in typename[1:]:
                if c not in OK_VARNAME_SUBSEQUENT:
                    warnings.append((4,'typename',typename,'c',c, 'not in',OK_VARNAME_SUBSEQUENT))
                    self.sanitizable = False
                    t = renpy.python.deleted
        oldt = None
        if self.sanitizable:
            oldt = None
            for mod in SRO_MODULES:
                t = getattr(mod, typename, None)
                if t:
                    if oldt and (t is not oldt):
                        warnings.append((5, 'Type', typename, 'appears in multiple modules.'))
                        self.sanitizable = False
                    else:
                        oldt = t
            if not oldt:
                warnings.append((4, 't is false'))
                self.sanitizable = False
        if not self.sanitizable:
            warnings.append((5, 'Discarded non-sanitizable SRO type name', typename))
            typename = '- discarded - '+typename
        if (type(oldt) is not type(SanitizableRevertableObject)) or (SanitizableRevertableObject not in oldt.mro()):
            self.sanitizable = False
            warnings.append((5,typename,'is not a SanitizableRevertableObject type'))
        self.typename = typename
        self.l = []
    def handle(self, context_stack, line):
        self.last_lineno = line.num
        if line=='end':
            context_stack.pop()
            return
        elif line.startswith('id '):
            if not self.sanitizable:
                # Remove the return to allow non-scrubbable SROs to become
                # renpy.python.deleted instances in the memo dict - however,
                # I haven't really tested this so I don't know the full
                # implications.
                warnings.append((5,"Discarded non-sanitizable object id",line))
                if not INSECURE:
                    return
            memoctx = get_stack_memo_context(context_stack)
            self.objid = i64dec(line[3:])
            if self.objid in memoctx.OBJECT_MEMO:
                warnings.append((5,'ERROR: cloned object',self.typename,self.objid,tuple(context_stack)))
            memoctx.OBJECT_MEMO[self.objid] = self
            return
        self.l.append(parse_tainted_tnv(line, context_stack))

    def lookup_sro_type(self):
        found = False
        for mod in SRO_MODULES:
            t = getattr(mod, self.typename, None)
            if t:
                if found and (t is not found):
                    warnings.append((5,"Duplicated type name found."))
                    return False
                found = t
        if not found:
            warnings.append((5, "Could not find sro type for", self.typename))
        if type(found) is not SanitizableRevertableObject and not issubclass(found, SanitizableRevertableObject):
            return False
        return found

    def deindirect(self):
        if self.actual_object is not None:
            return self.actual_object
        elif not self.sanitizable:
            self.actual_object = renpy.python.deleted
            warnings.append((5,'Deindirection of bogus type',self.typename,'returning deleted'))
            return renpy.python.deleted
        ret_type = self.lookup_sro_type()
        if not ret_type:
            self.actual_object = None
            warnings.append((5, "Found something that wasn't a SanitizableRevertableObject, discarded."))
            return None

        # We use __new__ instead of __init__ because a sanitizable revertable
        # object constructor takes an unknown number of arguments.
        ret = ret_type.__new__(ret_type)
        self.actual_object = ret
        warnings.note(ret_type.__name__)
        for item in self.l:
            # This will crash if the sanitizer does not exist, so is safe.
            warnings.note(item.name)
            try:
                setattr(ret, item.name, ret._member_sanitizers_[item.name](deindirect_if_needed(item.val)))
            except Exception, e:
                warnings.append((5, "Discarded non-deindirectable member", item.name, 'scrubber was', ret._member_sanitizers_.get(item.name,'ABSENT'),'val',item.val,deindirect_if_needed(item.val),e))
            warnings.unnote()
        ret.AfterLoad()
        warnings.unnote()
        return ret

DIRECT_TYPES = (bool, float, int, long, type(None), str, unicode, tuple,
    xlat.translated,
    renpy.python.RevertableList,)

def deindirect_if_needed(v):
    if isinstance(v, DeindirectableContext):
        return v.deindirect()
    if isinstance(v,SanitizableRevertableObject):
        return v
    if type(v) not in DIRECT_TYPES:
        raise Exception("Unexpected type "+str(type(v)))
    return v

class SafeloadTupleValueContext(SafeloadSequenceValueContext):
    def __init__(self):
        self.l = []
    def handle(self, context_stack, line):
        if line == ')':
            context_stack.pop()
            return
        self.l.append(unserialize_value(context_stack, line))
    def deindirect(self):
        ret = tuple([deindirect_if_needed(i) for i in self.l])
        return ret

class SafeloadRevertableSetValueContext(SafeloadSequenceValueContext):
    def handle(self, context_stack, line):
        if line==')':
            context_stack.pop()
            return
        self.l.append(unserialize_value(context_stack, line))
    def deindirect(self):
        return renpy.python.RevertableSet([deindirect_if_needed(i) for i in self.l])

class SafeloadPythonSetValueContext(SafeloadSequenceValueContext):
    def handle(self, context_stack, line):
        if line==')':
            context_stack.pop()
            return
        self.l.append(unserialize_value(context_stack, line))
    def deindirect(self):
        return set([deindirect_if_needed(i) for i in self.l])

PERMITTED_SCRIPT_LOCATION_DIRS = (
    # These are directories where script locations can be loaded from (e.g.
    # call stack, current location, return destinations.)
    '/home/renpy6-99-12/projects-6.99.12.2/WolfHall/game',
    '/home/renpy699/projects/Spaaaace/game',
    '/home/renpy699/projects/WolfHall/game',
    '/home/renpy/projects/WolfHall/game',
    '/home/renpy/projects/WolfHall-Demo/game',
    '/home/renpy/projects-6.99.12.2/WolfHall/game',
    '/home/renpy6-99-12/projects/WolfHall/game',
    )

def unserialize_script_loc(line):
    loc = line.split(' ',2)
    suffix = ()
    if loc[0] in ALLOWED_LOCATION_4TUPLE_SUFFICES:
        suffix = (loc[0],)
        loc = line.split(' ',3)[1:]
    if loc[0]=='lbl':
        if len(loc)!=2:
            warnings.append((5, 'Discarded unhandled script location',loc))
            return 'nopickles_replacement_label'
        loc = unescape_str(loc[1])
        if loc in ALLOWED_LOCATION_LABELS:
            return loc
        warnings.append((5, 'Discarded unhandled script location',loc))
        return 'nopickles_replacement_label'
    # Provisional sanity checking of locations in as of 2015 Apr 16.
    locsplit = loc[2].rsplit('/',1)
    if locsplit[0] not in PERMITTED_SCRIPT_LOCATION_DIRS:
        if locsplit[0] == '/home/tom/ab/renpy/renpy/common':
            # This stops saves from putting locations from the standard Ren'Py
            # library onto the call stack.  This is good because it saves me
            # from having to security-audit all the rpys in common/.
            #
            # nopickles_replacement_label will give a user-visible warning
            # if the game ever actually jumps there; we don't add a note
            # to the on-load warnings[] list because Ren'Py places stdlib
            # locations in unreachable-but-saved places on the call stack
            # sometimes.
            if AUTOHAMMER:
                # Wrapped in autohammer because a normal save will have
                # a bunch of these.
                print "Debug: unserialize_script_loc got disallowed",repr(loc)
            return 'nopickles_replacement_label'
        warnings.append((5,"Discarded unexpected script location", loc, locsplit))
        return 'nopickles_replacement_label'
    return (loc[2], int(loc[1]), int(loc[0])) + suffix

ALLOWED_LOCATION_LABELS = set((
    'nopickles_replacement_label',
    ))

SILENT_DISCARD_LOCATION_LABELS = set((
    '_call__load_reload_game_1',
    ))

class SafeloadCallstackContext(object):
    def __init__(self):
        self.call_location_stack = []
        self.return_stack = []
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
            return
        elif line.startswith('ret '):
            self.return_stack.append(unserialize_script_loc(line[4:]))
        elif line.startswith('retlbl '):
            if INSECURE:
                # XXX need to scrub retlbls.
                warnings.append((6,'accepted',line))
                self.return_stack.append(unescape_str(line[7:]))
            elif line[7:] in ALLOWED_LOCATION_LABELS:
                self.return_stack.append(line[7:])
            elif line[7:] in SILENT_DISCARD_LOCATION_LABELS:
                self.return_stack.append('nopickles_stack_error')
            else:
                # We assume all labels in game/ are safe to return to.
                lbl = None
                try:
                    lbl = renpy.game.script.lookup(line[7:])
                    if lbl.filename.startswith('game/'):
                        self.return_stack.append(line[7:])
                        return
                except renpy.script.ScriptError, e:
                    warnings.append((5,"Could not find label", line))
                    self.return_stack.append("nopickles_stack_error")
                    return
                warnings.append((5,'discarded',line,lbl.filename))
                unroller.traceback_include_fileinfo = True
                self.return_stack.append('nopickles_stack_error')
        else:
            self.call_location_stack.append(unserialize_script_loc(line))

def scrub_g_points(s):
    ret = renpy.python.RevertableDict()
    if type(s) is renpy.python.StoreDeleted:
        return renpy.python.deleted
    if type(s) is not renpy.python.RevertableDict:
        warnings.append((5,"Expected a RevertableDict, got",type(s)))
        return ret
    for k,v in s.iteritems():
        if type(k) is str and type(v) is int:
            ret[k] = v
    return ret

SCREEN_ARG_SCRUBBERS = sanitize_wh.SCREEN_ARG_SCRUBBERS

SCREEN_SCOPE_SCRUBBERS = sanitize_wh.SCREEN_SCOPE_SCRUBBERS

# Screen scope serializers are passed Name and Value and return a tnv.  This
# allows live datatypes to be used in screens as appropriate and revert to
# being strings or the like when not appropriate.
SCREEN_SCOPE_SERIALIZERS = sanitize_wh.SCREEN_SCOPE_SERIALIZERS

SERIALIZE_STORE_OVERRIDES = {
    # If present, the following constants will replace serialize_tnv for
    # the store (and only the store.)
    'store.registration_key' : '#',
    'registration_key' : '#',
    'store.updater_target_version' : ('# store.updater_target_version',),
    }
SERIALIZE_STORE_OVERRIDES.update(sanitize_wh.SERIALIZE_STORE_OVERRIDES)

STORE_SERIALIZERS = sanitize_wh.STORE_SERIALIZERS

STORE_SCRUBBERS = sanitize_wh.STORE_SCRUBBERS

STORE_SCRUBBERS.update(
    # Ren'Py internals.
    args = sanitize.expect_del_or_emptytuple,
    kwargs = sanitize.expect_del_or_emptydict,
    mouse_visible = sanitize.expect_bool,
    nvl_list = sanitize.scrub_nvl_list,
    save_name = sanitize.discard, # save_name variable doesn't need to be
                                  # preserved in saves, as the save name is
                                  # actually carried in SaveInfo()
    suppress_overlay = sanitize.expect_bool,
    _args = sanitize.expect_None,
    _defaults_set = sanitize.expect_del_or_rset_of_str,
    _game_menu_screen = sanitize.scrub__game_menu_screen,
    _history = sanitize.expect_bool_or_del,
    _history_list = sanitize.new_empty_revertable_list, # This is used for
    # Ren'Py's internal history, needs a proper scrubber if we are going to
    # actually use history but for Wolf Hall it can be disregarded for the nonce
    _kwargs = sanitize.expect_None,
    _last_raw_what = sanitize.expect_displayable_text_or_del_or_None,
    _last_say_what = sanitize.expect_displayable_text_or_None,
    _last_say_who = sanitize.scrub_last_say_who,
    _last_voice_play = sanitize.expect_del_or_None,
    _m1_developer__missing_scene = sanitize.expect_bool,
    _menu = sanitize.expect_bool_or_del,
    _old_game_menu_screen = sanitize.scrub__game_menu_screen,
    _old_history = sanitize.expect_bool_or_del,
    _old_predict_screens = sanitize.new_empty_revertable_list,
    _predict_screens = sanitize.new_empty_revertable_list, # Used in older Ren'Py
    _predict_screen = sanitize.new_empty_revertable_dict, # Used in 6.99.12.2
    _return = sanitize.expect_bool_or_int_or_None,
    _side_image_attributes = sanitize.scrub__side_image_attributes,
    _window_subtitle = sanitize.expect_empty_str,
    _windows_hidden = sanitize.expect_bool,

    # Spiky Ren'Py features
    caption = sanitize.expect_displayable_text_or_del_or_None,
    crash_ignored = sanitize.expect_bool,
    debugger_used = sanitize.expect_bool,
    http_result = sanitize.expect_str, # For traceback upload
    LAST_GPG_CTX = sanitize.always_del, # For traceback upload.
    load_count = sanitize.expect_del_or_int_or_None,
    old_load_count = sanitize.expect_del_or_int,
    updates_available = sanitize.new_empty_revertable_list,
    version_started_in = sanitize.expect_del_or_str,
    versions_played_in = sanitize.expect_del_or_set_of_str,
    # Used by save namer, no need to load.
    keyboard_target_widget = sanitize.discard,
    # message is used by the traceback uploader.
    message = sanitize.expect_displayable_text_or_del,
    old_key_repeat = sanitize.expect_del_or_int2, # Save namer
    save_caption_widget = sanitize.discard, # Save namer.
    warnstr = sanitize.discard, # Nopickles warnings.
    )

OLD_DATEWARP_STORE_SCRUBBERS = dict(
    # Deleted all the definitely-DW-specific stuff.
    current_save = sanitize.discard,
    result = sanitize.expect_del_or_str, # Used by traceback uploader
    predict_screens = sanitize.new_empty_revertable_list,
    ret = sanitize.expect_del_or_str,
    save_caption_widget = sanitize.expect_del,
    save_name = sanitize.expect_str,
    warnstr = sanitize.discard,
    _windows_hidden = sanitize.expect_bool,
    )
OK_VARNAME_SUBSEQUENT = set('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz')
OK_VARNAME_FIRST = set('ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz')

def scrub_varname(n):
    if n.startswith('__'):
        raise Exception("No replacing magic variables!")
    if n[0] not in OK_VARNAME_FIRST:
        raise Exception("Invalid character in variable name: "+n[0])
    for c in n[1:]:
        if c not in OK_VARNAME_SUBSEQUENT:
            raise Exception("Invalid character in variable name: "+c)
    return n

def scrub_tnv(line, scrubbers):
    tnv = line.split(' ',2)
    t = tnv[0]
    n = scrub_varname(tnv[1])
    # Singleton type/values first.
    if t=='None':
        rv = None
    elif t=='True':
        rv = True
    elif t=='False':
        rv = False
    elif t=='del':
        rv = renpy.python.deleted
    elif len(tnv)!=3:
        print "Debug 344:",line
        raise Exception("Malformed tnv line.")
    elif t=='str':
        rv = unescape_str(tnv[2])
    elif t=='xtr':
        rv = xlat.translated(unescape_str(tnv[2]))
    else:
        print "Flawed line was",line
        raise Exception("Unexpected tnv line.")
    if n in scrubbers:
        warnings.note('tnv '+n)
        ret = n, scrubbers[n](rv)
        warnings.unnote()
        return ret
    elif INSECURE:
        print "Debug: returning un-scrubbed entry for",line
        return n, rv
    print "No scrubber for",line
    raise Exception("No scrubber found.")

class SafeloadDynamicStackEntryContext(object):
    def __init__(self):
        self.stack = []
    def push(self):
        self.stack.append({})
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
            return
        warnings.note('dynamicstack')
        k, v = scrub_tnv(line, STORE_SCRUBBERS)
        warnings.unnote()
        self.stack[-1][k] = v

def unescape_music_filename(fn):
    unescaped = sanitize.scrub_relative_path(unescape_str(fn))
    if not unescaped.endswith('.ogg'):
        raise Exception('Bad music filename.')
    return unescaped

class SafeloadMusicChannelContext(object):
    secondary_volume = 1.0
    def __init__(self, name):
        for c in name:
            if c not in 'abcdefghijklmnopqrstuvwxyz0123456789':
                raise Exception("Disallowed characters in music channel name.")
        try: # We try integer channel names first.
            self.name = int(name)
        except:
            self.name = name
        self.filenames = []
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
        elif line.startswith('file '):
            self.filenames.append(unescape_music_filename(line[5:]))
        elif line.startswith('vol '):
            self.secondary_volume = float(line[4:])
        else:
            raise Exception("Unexpected line in music context.")

def scrub_store_identifier(s):
    if s=='store':
        return 'store'
    raise Exception("Unexpected store identifier "+s)

class SafeloadMusicContext(object):
    def __init__(self):
        self.channels = {}
    def handle(self, context_stack, line):
        if line.startswith('channel '):
            chan = SafeloadMusicChannelContext(line[8:])
            self.channels[chan.name] = chan
            context_stack.append(chan)
        elif line=='end':
            context_stack.pop()
        else:
            print "Line was",line
            raise Exception("Unexpected line in music context.")

class Tainted(object):
    def __init__(self, val, name=None):
        self.name = name
        self.val = val
    def deindirect_val(self):
        v = self.val
        if isinstance(v,DeindirectableContext):
            return v.deindirect()
        return v

def parse_tainted_tnv(line, context_stack):
    # This returns a tainted variable containing the line's value.
    # For values that create a new parsing context, the value will be the
    # parsing context.
    tnv = line.split(' ',2)
    if len(tnv)<2:
        raise Exception("Unhandled tnv "+line)
    t = tnv[0]
    n = tnv[1]
    if len(tnv)==3:
        v = tnv[2]
        if t=='int':
            return Tainted(int(v),n)
        elif t=='str':
            return Tainted(unescape_str(v),n)
        elif t=='float':
            return Tainted(float(v),n)
        elif t=='sro':
            ctx = SafeloadSanitizableRevertableObjectContext(v, lineno=line.num)
            context_stack.append(ctx)
            return Tainted(ctx, n)
        elif t=='sroref':
            # Find the memo.
            memoctx = get_stack_memo_context(context_stack)
            # And return the object from it.
            # If we get an error here whining about a KeyError in
            # parse_tainted_tnv, this can be fixed by changing the
            # SanitizableRevertableObjectContext to put non-sanitizable SROs
            # into the memo dict.
            return Tainted(memoctx.OBJECT_MEMO[i64dec(v)], n)
        elif t=='xform':
            if v not in transforms_by_name:
                warnings.append((5,"Invalid transform name",v,"substituting dispcenter"))
                return Tainted(transforms_by_name['dispcenter'], n)
            return Tainted(transforms_by_name[v], n)
    elif len(tnv)==2:
        if t=='False':
            return Tainted(False, n)
        elif t=='None':
            return Tainted(None, n)
        elif t=='True':
            return Tainted(True, n)
        elif t=='(':
            ctx = SafeloadTupleValueContext()
            context_stack.append(ctx)
            return Tainted(ctx, n)
        elif t=='[':
            ctx = SafeloadRevertableListValueContext()
            context_stack.append(ctx)
            return Tainted(ctx, n)
        elif t=='{':
            ctx = SafeloadRevertableDictValueContext()
            context_stack.append(ctx)
            return Tainted(ctx, n)
        elif t=='p{':
            ctx = SafeloadPythonDictValueContext()
            context_stack.append(ctx)
            return Tainted(ctx, n)
        elif t=='rs(':
            context_stack.append(SafeloadRevertableSetValueContext())
            return Tainted(context_stack[-1],n)
        elif t=='s(':
            context_stack.append(SafeloadPythonSetValueContext())
            return Tainted(context_stack[-1],n)
        elif t=='del':
            return Tainted(renpy.python.deleted, n)
    raise Exception('Unhandled tnv '+line)

class SafeloadStoreContext(object):
    def __init__(self):
        self.data = []
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
            return
        self.data.append(parse_tainted_tnv(line, context_stack))

class SafeloadObjectsContext(object):
    onval = False
    def __init__(self):
        self.trios = []
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
            return
        elif self.onval:
            self.trios[-1].append(unserialize_value(context_stack, line))
            self.onval = False
            return
        self.trios.append(line.split(' ',1))
        self.onval = True
        

class SafeloadRollbackLogEntryContext(object):
    scene = None
    dynamicstack = None
    music = None
    def __init__(self):
        self.stores = {}
        self.tainted_roots = []
        self.OBJECT_MEMO = {}

    def resolve_tainted_roots(self):
        self.untainted_roots = {}
        for t in self.tainted_roots:
            if t.name.startswith('store.'):
                varname = t.name[6:]
            else:
                raise Exception("Unexpected store variable name.")
            if varname in STORE_SCRUBBERS:
                warnings.note(('tr',varname))
                self.untainted_roots[t.name] = STORE_SCRUBBERS[varname](t.deindirect_val())
                warnings.unnote()
            elif INSECURE:
                warnings.append((6,"Accepted non-scrubbable variable",varname,t.deindirect_val()))
                self.untainted_roots[t.name] = t.deindirect_val()
            else:
                warnings.append((5,'No scrubber for '+varname+', discarded it.',t))
    def handle(self, context_stack, line):
        if line=='end':
            self.resolve_tainted_roots()
            context_stack.pop()
            return
        elif line=='callstack':
            context_stack.append(SafeloadCallstackContext())
            self.callstack = context_stack[-1]
            return
        elif line.startswith('current '):
            self.current = unserialize_script_loc(line[8:])
            return
        elif line=='dynamic':
            if self.dynamicstack is None:
                self.dynamicstack = SafeloadDynamicStackEntryContext()
            self.dynamicstack.push()
            context_stack.append(self.dynamicstack)
            return
        elif line=='music':
            self.music = SafeloadMusicContext()
            context_stack.append(self.music)
            return
        #elif line=='objects':
        #    self.objects = SafeloadObjectsContext()
        #    context_stack.append(self.objects)
        #    return
        elif line=='scene':
            if self.scene is not None:
                raise Exception("RollbackLogEntry already had a scene.")
            context_stack.append(SafeloadSceneContext())
            self.scene = context_stack[-1]
            return
        elif line.startswith('store '):
            storename = scrub_store_identifier(line[6:])
            context_stack.append(SafeloadStoreContext())
            self.stores[storename] = context_stack[-1]
            return
        self.tainted_roots.append(parse_tainted_tnv(line, context_stack))

    OK_MUSIC_CHANNELS = set((7, 'music2','music3','pos1','pos2','pos3','sound','sound2','sound3'))
    def music_update_data(self):
        if not self.music:
            return ()
        global last_music_update_data
        ret = {}
        for channelname, data in self.music.channels.iteritems():
            if channelname in self.OK_MUSIC_CHANNELS:
                ctx = renpy.audio.audio.MusicContext()
                ctx.last_filenames = data.filenames
                ctx.secondary_volume = data.secondary_volume
                ctx.last_changed = renpy.audio.audio.get_serial()
                ret[channelname] = ctx
            else:
                print "Debug 879:",channelname,data
                raise Exception("Invalid channel in save.")
            last_music_update_data = ret
        return ret
    def stores_update_data(self):
        ret = {}
        for storek, storev in self.stores.iteritems():
            if storek!='store':
                raise Exception("Cannot store stores other than store.")
            ret[storek] = r = dict()
            for item in storev.data:
                n = item.name
                if n in STORE_SCRUBBERS:
                    warnings.note('stores update '+n)
                    r[n] = STORE_SCRUBBERS[n](item.deindirect_val())
                    warnings.unnote()
                elif INSECURE:
                    warnings.append((6,"No scrubber for",n,item.val))
                    r[n] = item.deindirect_val()
                else:
                    warnings.append((5,'Discarded store item',n,'because no scrubbers findable.'))
        return ret

class SafeloadRollbackLogContext(object):
    def __init__(self):
        self.entries = []
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
        elif line=='entry':
            context_stack.append(SafeloadRollbackLogEntryContext())
            self.entries.append(context_stack[-1])
        else:
            raise Exception("Unexpected line in RollbackLog context.")

# The loader works on a context stack.  Context zero is the outermost.

class SafeloadContextZero(object):
    scene = None
    log = None
    info = None
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
        elif line=='log':
            if self.log is not None:
                raise Exception("Context zero already had a log.")
            context_stack.append(SafeloadRollbackLogContext())
            self.log = context_stack[-1]
        elif line=='saveinfo':
            context_stack.append(SaveInfo())
            self.info = context_stack[-1]
        elif line:
            print "Context zero rejecting line '"+line+"'"
            raise Exception("Inconsistent save.")
        # We ignore blank lines, to make savegame editing less annoying.

LOADABLE_SCREENS = set((
    # _missing_images is believed unnecessary, as it crashes on load unless I
    # save contexts, and I don't want to save contexts.
    'black_readout',
    'blue_readout',
    'dungeon',
    'effect_tab',
    'green_readout',
    'glowies',
    'merit_readout',
    'minimap',
    'money_readout',
    'readout_button',
    'red_readout',
    'seniors_yelling',
    #'skip_indicator',
    'smart_readout',
    'spellcard',
    'spring_break_text',
    'statbar',
    'stress_readout',
    'strong_readout',
    'timestamp',
    'timestampish',
    'white_readout',
    'winter_holiday_text',
    ))

DISCARDABLE_ATS = set((
    'cardcentered', # Has animation params, we don't actually need them.
    ))

DISCARDABLE_SCREENS = set((
    '_autoreload', # Autoreload is a dev tool, we don't really need it in
                    # saves.
    '_missing_images', # Crashes on load unless I save contexts.  I don't want
                        # to save contexts.
    'checklist_notify', # The checklist notification screen is transient.
    'cmd_target_known_cell',
    'cmd_targetlist',
    'choose_partymember_action', # Transient.
    'dungeon_help', 
    'exit_game_button', # Only used at the end of the demo, has special fixups
                        # around it anyhow.
    'notify', # The Ren'Py notification screen is transient.
    'skip_indicator', # That should be controlled by the player
    'spelltarget',
    'trophy_unlock', # This is transient.
    ))

class SafeloadSceneContext(object):
    def __init__(ctx):
        ctx.layers = []
    def handle(ctx, context_stack, line):
        if line=='end':
            context_stack.pop()
        elif line.startswith('layer '):
            context_stack.append(SafeloadLayerContext(line[6:]))
            ctx.layers.append(context_stack[-1])
        else:
            print ctx,'got',line
            raise Exception("Inconsistent save.")
    def SceneLists(self):
        showing = renpy.display.image.ShownImageInfo()
        ret = renpy.display.core.SceneLists(None,showing)
        for layer in self.layers:
            for img in layer.images:
                if type(img) is not SafeloadScreenContext:
                    if img.ats:
                        at_list = img.ats
                    else:
                        at_list = [renpy.store.dispcenter]
                    if img.tag is None:
                        key = img.what
                    else:
                        key = img.tag
                    if img.tag.startswith('hide$'):
                        warnings.append((0,"Save contained a hiding displayable; discarded it."))
                        continue
                    what = img.what.split(' ')
                    if what[0]!=img.tag:
                        name = tuple([img.tag,]+what[1:])
                    else:
                        name = tuple(what)
                    ret.add(layer.name, renpy.easy.displayable(img.what), at_list=at_list, key=key, name=name)

                    showing.attributes[(layer.name, key)] = tuple(img.what.split(' '))
                else:
                    tag = img.__dict__.pop('tag')
                    if tag not in LOADABLE_SCREENS:
                        if tag in DISCARDABLE_SCREENS:
                            continue
                        elif INSECURE:
                            warnings.append((6,'Accepted screen of unknown loadability',tag,img.__dict__))
                        else:
                            warnings.append((5,'Discarded screen of unknown loadability',tag,img.__dict__))
                            continue
                    scope = img.__dict__.pop('scope',{})
                    img.__dict__.pop('name',None)
                    if img.__dict__:
                        warnings.append((5,'Screen: discarded __dict__',img.__dict__))
                    screen = renpy.display.screen.get_screen_variant(tag.split()[0])
                    try:
                        d = renpy.display.screen.ScreenDisplayable(screen, screen.tag, layer.name, widget_properties={}, scope=scope)
                        ret.add(layer.name, d, name=tag, key=tag)
                    except Exception,e:
                        warnings.append((5,'An exception occurred loading a screen, it has been discarded',tag,e))
                        continue
        return ret
# add args: (SceneLists, layername, exact_displayable, key=tag
#print "Debug:add(",self,layer,thing,key,zorder,behind,at_list,name,atl
# Debug:add( <renpy.display.core.SceneLists object at 0x373cf90> master <ATLTransform at 7fd7d9a60790> bg 0 [u'vn_overlay'] [<NamedTransform overlay_pos 0x370bf90>] (u'bg', u'rainroad') None


class SafeloadLayerContext(object):
    saybehavior = False
    def __init__(ctx, name):
        ctx.name = name
        ctx.images = [] # Also contains any screens.
    def handle(ctx, context_stack, line):
        if line=='end':
            context_stack.pop()
        elif line.startswith('img '):
            context_stack.append(SafeloadImageContext(unescape_str(line[4:])))
            ctx.images.append(context_stack[-1])
        elif line.startswith('screen '):
            context_stack.append(SafeloadScreenContext(line[7:]))
            ctx.images.append(context_stack[-1])
        elif line=='saybehavior':
            ctx.saybehavior = True
        else:
            raise Exception("Inconsistent save.")

def scrub_plainstr(s):
    for c in s:
        if c not in 'abcdefghijklmnopqrstuvwxyz':
            print "Debug:",s
            raise Exception("Invalid plainstr")
    return s

def scrub_accept_anything(s):
    if INSECURE:
        warnings.append((6,'scrub_accept_anything got',type(s),s))
        return s
    raise Exception("scrub_accept_anything only allowed in insecure mode.")

class SafeloadScreenContext(object):
    ats = None
    name = None
    # vestiges of the old sanitizers.  Now uses SCREEN_SCOPE_SCRUBBERS
    # instead.
    # sanitizable.
    #    char = scrub_accept_anything,
    #    side_image = scrub_accept_anything,
    #    what = scrub_what,
    #    what_id = scrub_plainstr,
    #    who = scrub_accept_anything,
    #    who_id = scrub_plainstr,
    #    window_id = scrub_plainstr,
    #    )
    def __init__(ctx, name):
        ctx.tag = name
        ctx.scope = dict()
    def handle(ctx, context_stack, line):
        if line=='end':
            if ctx.name in SCREEN_ARG_SCRUBBERS:
                ctx.scope['_args'], ctx.scope['_kwargs'] = SCREEN_ARG_SCRUBBERS[ctx.name](ctx.name, ctx.scope)
            context_stack.pop()
        elif line.startswith('name '):
            ctx.name = line[5:]
        elif line.startswith('None '):
            varname = line[5:]
            if ctx.name in SCREEN_SCOPE_SCRUBBERS:
                sss = SCREEN_SCOPE_SCRUBBERS[ctx.name]
                if varname in sss:
                    ctx.scope[varname] = sss[varname](None)
                    return
            warnings.append((5,"Discarded unscrubbable screen scope var",ctx.name,varname,None))
        elif line.startswith('str '):
            varname, val = line[4:].split(' ',1)
            if ctx.name in SCREEN_SCOPE_SCRUBBERS:
                sss = SCREEN_SCOPE_SCRUBBERS[ctx.name]
                if varname in sss:
                    ctx.scope[varname] = sss[varname](unescape_str(val))
                    return
            warnings.append((5,"Discarded unscrubbable screen scope var",ctx.name,line))
        else:
            raise Exception("Inconsistent save.")

class SafeloadImageContext(object):
    def __init__(ctx, tag):
        ctx.tag = tag
        ctx.ats = []
    def handle(ctx, context_stack, line):
        if line.startswith('at '):
            ctx.ats = []
            for atname in line[3:].split(','):
                trans = getattr(renpy.store, atname, None)
                if type(trans) is NamedTransform:
                    ctx.ats.append(trans)
                elif trans is None:
                    warnings.append((5,"Unexpected transform name",atname))
                else:
                    warnings.append((5,"Unexpected transform type",type(trans)))
        elif line=='end':
            context_stack.pop()
        elif line.startswith('what '):
            ctx.what = unescape_str(line[5:])
        else:
            raise Exception("Inconsistent save.")

class NumberedStr(str):
    num = 0

def safeload(lines):
    zero = SafeloadContextZero()
    context_stack = [zero]

    lineno = 0
    for i in range(0,len(lines)):
        lines[i] = NumberedStr(lines[i])
        lines[i].num = i+1
    for line in lines:
        lineno += 1

        if line.startswith('#'):
            continue
        try:
            context_stack[-1].handle(context_stack, line)
        except Exception,e:
            print "Stack was",context_stack
            print "Line was",lineno,line
            print "Exception was",e
            raise

    sls = renpy.exports.scene_lists()
    for layer in sls.layers:
        sls.layers[layer] = []
        sls.at_list[layer] = dict()

    newlog = renpy.python.RollbackLog()
    if zero.log is None:
        raise Exception("Load error: No rollback log in save?")
    for ent in zero.log.entries:
        newent = renpy.python.Rollback()
        newlog.log.append(newent)
        newent.context = renpy.execution.Context(True)
        newent.context.runtime = zero.info.playtime
        newent.context.call_location_stack = list(ent.callstack.call_location_stack)
        newent.context.return_stack = list(ent.callstack.return_stack)
        newent.context.current = ent.current
        newent.context.scene_lists = ent.scene.SceneLists()
        newent.context.images = newent.context.scene_lists.shown
        newent.context.dynamic_stack = ent.dynamicstack.stack
        newent.context.music.update(ent.music_update_data())
        newent.checkpoint = True
        for k,v in ent.stores_update_data().iteritems():
            if k in newent.stores:
                raise Exception("Redefining an existing store.")
            newent.stores[k] = v
    if newlog.log:
        newlog.current = newlog.log[-1]
        roots = zero.log.entries[-1].untainted_roots
    try:
        unroller.last_save_caption = zero.info.caption
    except:
        unroller.last_save_caption = ''
    try:
        unroller.save_started_in_version = zero.info.started_in_version
        unroller.save_played_in_versions = zero.info.played_in_versions
    except:
        unroller.save_started_in_version = 'unknown'
        unroller.save_played_in_versions = ['unknown',]
    print "Debug: unroller.last_save_caption now",unroller.last_save_caption
    unroller.debugger_used = unroller.debugger_used or zero.info.debugger_used
    unroller.crash_ignored = unroller.crash_ignored or zero.info.crash_ignored
    currentver = renpy.store.slurpfile('game_version')[1][0]
    if currentver not in unroller.save_played_in_versions:
        unroller.save_played_in_versions.append(currentver)
    if DEBUG:
        # Keep this around for poking.
        global last_zero
        last_zero = zero
        global last_newlog
        last_newlog = newlog
    global last_loaded_game_info
    last_loaded_game_info = zero.info
    newlog.unfreeze(roots, label='_after_load')
    raise Exception("Unfreeze failed.")

    renpy.exports.restart_interaction()

def asciize(d,recursion=0):
    ret = ''
    if isinstance(d,dict):
        if recursion>3:
            return 'Excess recursion.'
        for k, v in d.iteritems():
            ret += asciize(k,recursion+1)+':'+str(type(v))+'='+asciize(v,recursion+1)+';'
    elif type(d) is bool:
        return str(d)
    else:
        try:
            return str(d)
        except:
            return "<could not asciify "+str(type(d))+">"
    return ret

def escape_str(s):
    ret = '';
    if type(s) is not str and type(s) is not unicode and type(s) is not xlat.translated:
        raise Exception("escape_str called with invalid type "+str(type(s)))
    for c in s:
        if c=='\\':
            ret += '\\\\'
        elif c=='\n':
            ret += '\\n'
        else:
            oc = ord(c)
            if oc>=32 and oc<=126:
                ret += c
            elif oc<256:
                ret += "\\x%02x"%oc
            else:
                ret += "\\U%x;"%oc
    return str(ret)

def unescape_str(s):
    ret = ''
    i = 0
    while i<len(s):
        c = s[i]
        if c!='\\':
            ret += c
        else:
            i += 1
            c = s[i]
            if c=='\\':
                ret += '\\'
            elif c=='n':
                ret += '\n'
            elif c=='x':
                ret += unichr(int(s[i+1:i+3],16))
                i += 2
            elif c=='U':
                a = ''
                i += 1
                while s[i] in '0123456789abcdef':
                    a += s[i]
                    i += 1
                ret += unichr(int(a,16))
        i += 1
    return ret

def append_at_lists(serialized, disp, at_list):
    ats = []
    if type(disp) is renpy.display.motion.ATLTransform:
        if disp.varname:
            ats.append(disp.varname)
        else:
            serialized.append('#.1490 no varname')
    else:
        serialized.append('#.1484 '+str(type(disp)))
    for at in at_list:
        if type(at) is NamedTransform:
            if ats and ats[-1]!=at.name:
                ats.append(at.name)
        else:
            unroller.last_unstringifiable_at = (at,at_list)
            if ats and ats[-1]!=at.varname:
                ats.append(at.varname)
    if ats:
        serialized.append('at '+(','.join(ats)))

def stringify_at_list(at_list):
    ret = []
    for at in at_list:
        if type(at) is NamedTransform:
            ret.append(at.name)
        else:
            unroller.last_unstringifiable_at = (at,at_list)
            ret.append(at.varname)
    return ','.join(ret)

def serialize_scene_lists(sls):
    ret = ['scene']
    for layername in sls.layers:
        ret.append('layer '+layername)
        layer = sls.layers[layername]
        ats = sls.at_list[layername]
        for shown in layer:
            if FUTUREDEBUG:
                try:
                    ret.append('#'+repr(shown)+str(type(shown.displayable)))
                except:
                    ret.append('#Unrepresentable'+str(type(shown.displayable)))
            if shown.tag is not None:
                if shown.tag.startswith('hide$'):
                    # So we don't wind up sticking a spurious 'end' after
                    # a hide$ displayable - which would be bad because
                    # it would close the layer.  Also first, because a
                    # hide$ can be a ScreenDisplayable.
                    continue
                elif type(shown.displayable) is renpy.display.screen.ScreenDisplayable:
                    if ' '.join(shown.name) in DISCARDABLE_SCREENS:
                        # Don't bother saving screens that we won't be loading.
                        if VERBOSE_SAVES:
                            ret.append("#.1766 "+(' '.join(shown.name)))
                        continue
                    ret.append('screen '+shown.tag)
                    if shown.name != shown.tag:
                        if type(shown.name) is str:
                            shown_name = shown.name
                            ret.append('name '+shown_name)
                        elif type(shown.name) is tuple:
                            shown_name = ' '.join(shown.name)
                            ret.append('name '+shown_name)
                        else:
                            shown_name = None
                            print "WARNING: Non-string ScreenDisplayable name?",shown.name
                            ret.append('name '+' '.join(shown.name))
                            ret.append('#.1573 BUG:'+str(type(shown.name))+str(shown.name))
                    else:
                        # We always include the name so the scrubbers will get
                        # called.
                        shown_name = shown.tag
                        ret.append('name '+shown.name)
                    for k,v in shown.displayable.scope.iteritems():
                        if shown_name in SCREEN_SCOPE_SERIALIZERS and k in SCREEN_SCOPE_SERIALIZERS[shown_name]:
                            ret.extend(SCREEN_SCOPE_SERIALIZERS[shown_name][k](k, v))
                        elif type(v) is str or type(v) is unicode:
                            ret.append('str '+k+' '+escape_str(v))
                        elif v is None:
                            ret.append('None '+k)
                        else:
                            ret.append('# Skipped encoding '+k+' '+str(type(v)))
                elif not shown.tag.startswith('hide$'):
                    # Don't save hiding anims.
                    ret.append('#.1521')
                    ret.append('img '+escape_str(shown.tag))
                    if shown.orig_name is not None:
                        if type(shown.orig_name) is str or type(shown.orig_name) is unicode:
                            ret.append(str('what '+escape_str(shown.orig_name)))
                        else:
                            ret.append('what '+escape_str(' '.join(shown.orig_name)))
                    elif shown.name is not None:
                        ret.append(('what '+escape_str(' '.join(shown.name))))
                    else:
                        ret.append("#.688")
                    if shown.orig_name is not None and shown.orig_name!=shown.name:
                        ret.append('#.1238 '+(' '.join(shown.name)).encode('utf-8','replace'))
                append_at_lists(ret, shown.displayable, ats[shown.tag])
                #if shown.tag in ats and ats[shown.tag]:
                #    ret.append('at '+stringify_at_list(ats[shown.tag]))
                #    #ret.append(stringify_at_block(ats[shown.tag]))
                #else:
                #    if type(shown.displayable) is renpy.display.motion.ATLTransform:
                #        ret.append('#.1536'+shown.displayable.varname)
                #    ret.append('#No ats')
                ret.append('end')
            elif type(shown.displayable) is renpy.display.behavior.SayBehavior:
                ret.append('saybehavior')
            else:
                print "Debug: discarded scene entry with tag None",shown
        ret.append('end')
    ret.append('end')
    ret.append('#/scene')
    return ret

def serialize_black_closet_save_info():
    ret = ['saveinfo']
    try:
        if unroller.last_save_caption:
            ret.append('caption '+escape_str(unroller.last_save_caption))
    except:
        pass
    try:
        if renpy.store.Firstname and renpy.store.Lastname:
            ret.append('name '+escape_str(renpy.store.Firstname+' '+renpy.store.Lastname))
    except:
        pass
    ret.append('month '+str(renpy.store.monthnum))
    ret.append('day '+str(renpy.store.day))
    try:
        ret.append('startver '+renpy.store.version_started_in)
    except:
        ret.append('#startver exception.')
    try:
        ret.append('playvers '+(';'.join(renpy.store.versions_played_in)))
    except:
        ret.append('#playvers exception.')
    ret.append('playtime '+str(int(renpy.exports.get_game_runtime())))
    if unroller.crash_ignored:
        ret.append('crash_ignored')
    if unroller.debugger_used:
        ret.append('debugger_used')
    ret.append('# end saveinfo')
    ret.append('end')
    return ret

def safesave_serialize():
    global serialize_warnings
    if DEBUG:
        serialize_warnings = []
    ret = serialize_black_closet_save_info()+serialize_rollback_log(renpy.game.log)
    return ret

def NewestSave():
    newest = None
    newest_mtime = 0
    for fn in os.listdir(renpy.config.savedir):
        if fn.endswith('.whs'):
            try:
                fsplit = fn[:-4].split('-',1)
                if fsplit[0]=='auto':
                    continue
                int(fsplit[0])
                int(fsplit[1])
                st = os.stat(os.path.join(renpy.config.savedir, fn))
                if st.st_mtime>newest_mtime:
                    newest = fn
                    newest_mtime = st.st_mtime
            except Exception,e:
                # Just quietly ignore saves that we can't figure out, they're
                # probably irrelevant.
                pass
    try:
        ns = newest.split('-')
        return (ns[0], int(ns[1][:-4]), newest_mtime)
    except:
        return None

def NewestAutoSave():
    newest = None
    newest_mtime = 0
    for fn in os.listdir(renpy.config.savedir):
        if fn.endswith('.whs'):
            try:
                fsplit = fn[:-4].split('-',1)
                if fsplit[0]!='auto':
                    continue
                int(fsplit[1])
                st = os.stat(os.path.join(renpy.config.savedir, fn))
                if st.st_mtime>newest_mtime:
                    newest = fn
                    newest_mtime = st.st_mtime
            except Exception,e:
                print 1697,e,fn
    try:
        ns = newest.split('-')
        return (ns[0], int(ns[1][:-4]), newest_mtime)
    except:
        return None

saveinfo_cache = dict()
def get_save_caption(slotname):
    ret = saveinfo_cache.get(slotname, None)
    if ret:
        return ret.caption or ''
    ret = SaveInfo()
    ret.read_from_file(os.path.join(renpy.config.savedir, slotname+'.whs'))
    saveinfo_cache[slotname] = ret
    return ret.caption or ''

class SaveInfoDecompressor(file):
    def __init__(self, f):
        self.f = f
        self.lines = []
    def readline(self):
        if not self.lines:
            decompresser = zlib.decompressobj()
            self.lines = decompresser.decompress(base64.b64decode(self.f.read(4096))).split('\n')
        return self.lines.pop(0)

class SaveInfo(object):
    caption = None
    mtime = None
    readable = False

    debugger_used = False
    crash_ignored = False

    # BC-specific
    month = None
    day = None

    name = None

    playtime = 0
    exists = False

    def __init__(self, slotnum=None):
        if slotnum:
            self.read_from_slotnum(slotnum)
    def read_from_slotnum(self, filename):
        try:
            fn = os.path.join(renpy.config.savedir,renpy.store.persistent._file_page+'-'+str(filename)+'.whs')
        except:
            return
        self.read_from_file(fn)
    def read_from_file(self, fn):
        warnings.note(("SaveInfo for",fn[-20:]))
        try:
            st = os.stat(fn)
            self.mtime = st.st_mtime
            f = open(fn, 'rb')
            line = True
            parse = False
            while line:
                line = f.readline()
                if line.endswith('\n'):
                    line = line[:-1]
                if line=='b64z':
                    f = SaveInfoDecompressor(f)
                    continue
                if line=='saveinfo':
                    parse = True
                elif not parse:
                    continue
                elif line=='end':
                    self.readable = True
                    warnings.unnote()
                    return
                else:
                    self.handle([None,],line)
            self.exists = True
        except Exception,e:
            pass
        warnings.unnote()
    def handle(self, context_stack, line):
        if line.startswith('caption '):
            # This assumes that caption is used in a context where Ren'Py
            # text tags are interpolated, but [variable] and %(variable)s
            # style tags are not.
            self.caption = unescape_str(line[8:]).replace('{','{{')
        elif line == 'crash_ignored':
            self.crash_ignored = True
        elif line == 'debugger_used':
            self.debugger_used = True
        elif line=='end':
            context_stack.pop()
        elif line.startswith('startver '):
            self.started_in_version = line[9:]
        elif line.startswith('playvers '):
            self.played_in_versions = line[9:].split(';')
        # Game-specific, for Magical Diary
        elif line.startswith('day '):
            self.day = int(line[4:])
        elif line.startswith('month '):
            self.month = int(line[6:])
        elif line.startswith('name '):
            self.name = sanitize.expect_displayable_text(line[5:])
        elif line.startswith('playtime '):
            self.playtime = int(line[9:])
        elif not line.startswith('#'):
            print "Debug: Unexpected savegame info line",line
    def InfoStr(self):
        if self.caption:
            if self.timer:
                return self.caption+'\n'+self.timer
            else:
                return self.caption
        elif self.timer:
            return self.timer
        return ''

autosave_serial = 0
def autosave_hammer(savedir):
    # For debugging, formerly autosave.
    global autosave_serial
    while True:
        try:
            f = open(os.path.join(savedir, 'auto', str(autosave_serial)+'.das'),'rb')
            autosave_serial += 1
            f.close()
        except:
            f = open(os.path.join(savedir, 'auto', str(autosave_serial)+'.das'),'wb')
            serialized = safesave_serialize()
            try:
                f.writelines('\n'.join(serialized))
            except:
                print "Debug: dumping serialization failure data to unroller.autosave_failure_data"
                unroller.autosave_failure_data = tuple(serialized)
                if DEBUG:
                    renpy.store.find_type_in_seq(unroller.autosave_failure_data,unicode)
                raise
            f.close()
            autosave_serial += 1
            return

def choose_autosave_slot():
    mtimes = {}
    for slot in '123456':
        try:
            st = os.stat(os.path.join(renpy.config.savedir,'auto-'+slot+'.whs'))
        except Exception,e:
            print "Debug 1813:",e
            return slot
        mtimes[slot] = st.st_mtime
    oldest_slot = '1'
    oldest_mtime = mtimes.pop('1')
    for slot,t in mtimes.iteritems():
        if t<oldest_mtime:
            oldest_mtime = t
            oldest_slot = slot
    return oldest_slot

def autosave_playervisible():
    slot = choose_autosave_slot()
    renpy.store.nopickle_save('auto-'+slot)

def try_all_saves(savedir):
    files = os.listdir(savedir)
    warnings.note(("try_all_saves(",savedir,")"))
    for fn in files:
        warnings.note((fn,))
        try:
            if fn.endswith('.das') or fn.endswith('.whs'):
                f = file(os.path.join(savedir,fn),'rb')
                lines = f.read().split('\n')
                f.close()
                safeload(lines)
        except renpy.game.RestartTopContext:
            pass
        except renpy.game.RestartContext:
            pass
        except Exception,e:
            warnings.append((0,"An exception occurred loading",fn,e))
        warnings.unnote()
    warnings.unnote()

def TEST():
    print "Debug: Test cleared",len(warnings)
    clear_warnings()
    try_all_saves(os.path.join(renpy.config.savedir,'auto'))
    print "...  and the warnings went to",len(warnings)
    global uwarnings
    uwarnings = []
    seen = set()
    for warning in warnings:
        try:
            if warning[:-2] not in seen:
                seen.add(warning[:-2])
                uwarnings.append(warning)
        except:
            uwarnings.append(warning)

def init_save_data():
    unroller.last_save_caption = ''
    ver = renpy.store.slurpfile('game_version')[0][1]
    print "Debug 1794:",ver
    unroller.save_played_in_versions = [ver,]
    partver = renpy.store.slurpfile('partial_update')
    origver = renpy.store.slurpfile('original_game_version')
    if partver:
        ver += ';part:'+partver[0][1]
    if origver:
        ver += ';orig:'+origver[0][1]
    unroller.save_started_in_version = ver

class SafeloadPersistentContext(object):
    preferences = None
    def __init__(self):
        self.data = []
    def handle(self, context_stack, line):
        if line=='end':
            context_stack.pop()
            return
        elif line=='_preferences':
            context_stack.append(SafeloadStoreContext())
            self.preferences = context_stack[-1]
            return
        else:
            self.data.append(parse_tainted_tnv(line, context_stack))
    
class Persistent(object):
    # Built-in values:
    install_beta_versions = None
    install_alpha_versions = None
    _preferences = None

    # XXX Sanity-check _achievements, _achievement_progress, _iap_purchases,
    # _update_last_checked, _character_volume, _placeholder_gender

    _ok_underscored = ('_achievement_progress', '_achievements',
        '_character_volume', 
        '_chosen', '_file_folder', '_file_page', '_file_page_name',
        '_iap_purchases', '_placeholder_gender',
        '_seen_audio', '_seen_ever',
        '_seen_images', '_seen_translates', '_set_preferences',
        '_style_preferences', 
        '_update_last_checked', '_virtual_size', '_voice_mute')

    # Replacement for the Ren'Py persistent object.
    def __getattr__(self, name):
        if name in self.__dict__:
            return self.__dict__[name]
        return None
    def __setattr__(self, name, val):
        if name.startswith('_') and name not in self._ok_underscored:
            warnings.append((5,"Persistent disallowed forbidden preference",name,val,type(val)))
        if name in self.__dict__ and self.__dict__[name] == val:
            # Don't bother marking things changed if the change didn't
            # do anything.  XXX this needs to correctly handle member
            # changes.
            return
        self.__dict__[name] = val
    def __getstate__(self):
        print "Persistent.getstate"
        return ()
    def _update(self):
        raise Exception("_update deprecated.")
        if self._preferences is None:
            self.__dict__['_preferences'] = renpy.preferences.Preferences()
        if not self._seen_ever:
            self.__dict__['_seen_ever'] = {}
        if not self._seen_images:
            self.__dict__['_seen_images'] = {}
        if not self._chosen:
            self.__dict__['_chosen'] = {}
        if not self._seen_audio:
            self.__dict__['_seen_audio'] = {}
        if self._changed is None:
            self.__dict__['_changed'] = {}
    def __init__(self):
        self.__dict__['_preferences'] = renpy.preferences.Preferences()
        self.__dict__['_seen_ever'] = {}
        self.__dict__['_seen_images'] = {}
        self.__dict__['_chosen'] = {}
        self.__dict__['_seen_audio'] = {}
        self.__dict__['_changed'] = {}
        self.__dict__['_seen_translates'] = set()

def persistent_serialize(persistent):
    ret = ['persistent']
    ret.extend(serialize_tnv('install_alpha_versions',persistent.install_alpha_versions,rollobjmemo=()))
    ret.extend(serialize_tnv('install_beta_versions',persistent.install_beta_versions,rollobjmemo=()))
    for k,v in persistent.__dict__.iteritems():
        if k=='_changed': # This is just a list of the times members were
            # changed last.  It shouldn't be serialized.
            pass
        elif k=='_preferences':
            ret.extend(persistent_serialize_preferences(v))
        else:
            ret.extend(serialize_tnv(k,v,rollobjmemo=()))
    ret.append('end')
    return ret
def persistent_serialize_preferences(_preferences):
    ret = ['_preferences']
    for pref,val in _preferences.__dict__.iteritems():
        ret.extend(serialize_tnv(pref, val,rollobjmemo=()))
    ret.append('end')
    return ret

def scrub_persistent_changed(s):
    if s:
        warnings.append((5,"scrub_persistent_changed discarded",s))
    return {}

def scrub_persistent_crypto(s):
    if s is None:
        return None
    if s=='gpg' or s=='plaintext':
        return str(s)
    if s is False:
        return False
    warnings.append((5,'scrub_persistent_crypto discarded',s))
    return None

def scrub_datewarp_persistent_chosen(s):
    # Just silently discard the chosen because it's not used in Date Warp.
    # For other games, it will be a dict of (locationtuple):menuoption pairs.
    #if s:
    #    warnings.append((5,'scrub_persistent_chosen discarded',type(s),s))
    return {}

def scrub_persistent_seen_ever(s):
    if type(s) is not dict:
        warnings.append((5,'scrub_persistent_seen_ever discarded',type(s),s))
        return {}
    ret = {}
    for k,v in s.iteritems():
        if v is True:
            if type(k) is str:
                ret[k] = True
            elif type(k) is tuple and len(k)==3 and type(k[0]) is str and type(k[1]) is int and type(k[2]) is int:
                ret[k] = True
            else:
                warnings.append((5,'scrub_persistent_seen_ever discarded',k,v))
        else:
            warnings.append((5,'scrub_persistent_seen_ever discarded',k,v))
    return ret

def ok_persistent_single_seen_image(k,v):
    if v is not True:
        return False
    if type(k) is not tuple:
        return False
    for ik in k:
        if type(ik) is not str:
            return False
    return True

def scrub_persistent_seen_images(s):
    ret = {}
    for k,v in s.iteritems():
        if ok_persistent_single_seen_image(k,v):
            ret[k] = v
        else:
            warnings.append((5,'scrub_persistent_seen_images discarded',k,v))
    return ret

def scrub_persistent_style_preferences(s):
    if s or type(s) is not renpy.python.RevertableDict:
        warnings.append((5,'scrub_persistent_style_preferences discarded',type(s),s))
    return renpy.python.RevertableDict()

def scrub_persistent_voice_mute(s):
    if type(s) is renpy.python.RevertableSet and not s:
        return renpy.python.RevertableSet()
    warnings.append((5,'scrub_persistent_voice_mute discarded',s))
    return renpy.python.RevertableSet()

def scrub_cgs(s):
    ret = set()
    for k in s:
        if type(k) is str:
            ret.add(sanitize.scrub_relative_path(k))
        else:
            warnings.append((5,"Discarded invalid checklist item.", k))
    return ret

def scrub_checklist(s):
    ret = set()
    for k in s:
        if type(k) is str:
            ret.add(k)
        else:
            warnings.append((5,"Discarded invalid checklist item.", k))
    return ret

def scrub_persistent_language(s):
    if s is None:
        return None
    ret = ''
    if type(s) is str:
        for c in s:
            if c not in alphalow:
                warnings.append((5,'persistent.language may only contain lowercase ASCII characters.'))
                return None
            ret += c
        return ret
    warnings.append((5,"Invalid value for persistent.language:",s))
    return None

PERSISTENT_SCRUBBERS = dict(
    # Spiky.
    crypto = scrub_persistent_crypto,
    extra_controls = sanitize.expect_bool_or_None,
    hide_updater = sanitize.expect_bool,
    install_alpha_versions = sanitize.expect_bool_or_None,
    install_beta_versions = sanitize.expect_bool_or_None,
    language = scrub_persistent_language,
    log = sanitize.expect_int_or_None,
    #logging = sanitize.expect_bool_or_None,
    use_visible_keyboard = sanitize.expect_bool,
    steam_achievements = sanitize.scrub_persistent_steam_achievements,
    steam_stats = sanitize.scrub_persistent_steam_stats,
    updater_url = sanitize.scrub_persistent_updater_url,

    # Magical Diary specific:
    bg_brightness = sanitize.expect_float_or_None,
    crop_widetalker = sanitize.expect_bool_or_None,
    codex_seen = sanitize.expect_set_of_str,
    codex_unlocked = sanitize.expect_set_of_str,
    dungeon_tileset = sanitize.scrub_relative_dirpath,
    spell_sort = sanitize.expect_int,

    # Trophies live here.
    checklist = scrub_checklist,

    # Black Closet specific:
    #allow_plotless = sanitize.expect_bool_or_None,
    #art_mode = sanitize.expect_int_or_None,
    #cgs = scrub_cgs,
    #easy = sanitize.expect_bool_or_None,
    #eggs = sanitize.expect_int_or_None,
    #hide_odds = sanitize.expect_bool_or_None,
    #loaded_dice = sanitize.expect_bool_or_None,    

    ## Datewarp-specific.
    #endings_reached = scrub_endings_reached,
    #wires_cheats = scrub_wires_cheats,

    # Stock Ren'Py prefs.  Note that some are forced blank because I don't
    # use them - usually because I've rolled my own similar code.
    _achievement_progress = sanitize.new_empty_python_dict,
    _achievements = sanitize.new_empty_python_set,
    _changed = scrub_persistent_changed,
    _character_volume = sanitize.scrub__character_volume,
    _chosen = scrub_datewarp_persistent_chosen,
    _file_folder = sanitize.expect_int,
    _file_page = sanitize.scrub__file_page,
    _file_page_name = sanitize.new_empty_python_dict, # Needs a proper sanitizer
    # if we support Ren'Py file save page naming.
    _iap_purchases = sanitize.new_empty_python_dict, # I'm not using IAP
    _invoke_main_menu = sanitize.expect_bool,
    _placeholder_gender = sanitize.new_empty_python_dict,
    _seen_audio = sanitize.scrub_persistent_seen_audio,
    _seen_ever = scrub_persistent_seen_ever,
    _seen_images = scrub_persistent_seen_images,
    _seen_translates = sanitize.scrub_persistent_seen_translates,
    _set_preferences = sanitize.expect_bool,
    _style_preferences = scrub_persistent_style_preferences,
    # Using spiky updater instead.
    _update_last_checked = sanitize.new_empty_python_dict,
    _virtual_size = sanitize.expect_int2,
    _voice_mute = scrub_persistent_voice_mute,
    )

PREFERENCE_SCRUBBERS = dict(
    _achievement_progress = sanitize.new_empty_python_dict,
    _placeholder_gender = sanitize.new_empty_python_dict,

    afm_after_click = sanitize.expect_bool,
    afm_enable = sanitize.expect_bool,
    afm_time = sanitize.expect_float_or_int,
    desktop_rollback_side = sanitize.scrub_preference_rollback_side,
    emphasize_audio = sanitize.expect_bool,
    fullscreen = sanitize.expect_bool,
    joymap = sanitize.scrub_preference_joymap,
    joy_deadzone = sanitize.expect_float,
    language = sanitize.expect_None,
    mobile_rollback_side = sanitize.scrub_preference_rollback_side,
    mouse_move = sanitize.expect_bool,
    mute = sanitize.scrub_preference_mute,
    pad_enabled = sanitize.expect_bool, # Gamepad enabled y/n?
    performance_test = sanitize.expect_bool,
    physical_size = sanitize.scrub_preference_physical_size,
    renderer = sanitize.scrub_preference_renderer,
    self_voicing = sanitize.expect_bool,
    show_empty_window = sanitize.expect_bool,
    skip_after_choices = sanitize.expect_bool,
    skip_unseen = sanitize.expect_bool,
    transitions = sanitize.expect_int,
    text_cps = sanitize.expect_int,
    using_afm_enable = sanitize.expect_bool,
    video_image_fallback = sanitize.expect_bool,
    virtual_size = sanitize.expect_int2_or_None,
    voice_sustain = sanitize.expect_bool,
    volumes = sanitize.scrub_preference_volumes,
    wait_voice = sanitize.expect_bool,
    )
def scrub_persistent_preferences(prefdata):
    ret = renpy.preferences.Preferences()
    for item in prefdata.data:
        if item.name in PREFERENCE_SCRUBBERS:
            setattr(ret, item.name, PREFERENCE_SCRUBBERS[item.name](deindirect_if_needed(item.val)))
        else:
            warnings.append((5,'Discarded unhandled pref',item.name,item.val))
    return ret

def persistent_init():
    try:
        os.stat(os.path.join(renpy.config.savedir,'persistent.dat'))
    except:
        return Persistent()
    return persistent_load(os.path.join(renpy.config.savedir,'persistent.dat'))

def persistent_load(filename):
    if not filename.endswith('.dat'):
        print "Disabled persistent",filename
        return None
    f = open(filename,'rb')
    if f.readline() != 'persistent\n':
        warnings.append((5,"Could not load persistent: bad magic."))
        return Persistent()
    pctx = SafeloadPersistentContext()
    context_stack = [pctx]
    lineno = 1
    for line in f.readlines():
        if line.startswith('#'):
            continue
        if line.endswith('\n'):
            line = line[:-1]
        line = NumberedStr(line)
        line.num = lineno
        lineno += 1
        context_stack[-1].handle(context_stack, line)
        if not context_stack:
            break
    ret = Persistent()
    global last_pctx
    last_pctx = pctx
    for item in pctx.data:
        if item.name not in PERSISTENT_SCRUBBERS:
            # BlackCloset-specific
            #if item.name.startswith('scene_') and deindirect_if_needed(item.val) is True:
            #    setattr(ret, item.name, True)
            #    continue
            warnings.append((5,"Discarded unhandled persistent item",item.name,item.val))
        else:
            warnings.note('persistent '+item.name)
            val = PERSISTENT_SCRUBBERS[item.name](deindirect_if_needed(item.val))
            setattr(ret, item.name, val)
            warnings.unnote()
    ret.__dict__['_preferences'] = scrub_persistent_preferences(pctx.preferences)
    if not ret._seen_translates:
        ret._seen_translates = set()
    return ret

PERSISTENT_BACKUP_COUNT=15
def persistent_cycle():
    i = PERSISTENT_BACKUP_COUNT
    try:
        os.unlink(os.path.join(config.savedir, 'persistent.bak'+str(i)))
    except:
        pass
    while i>0:
        i -= 1
        try:
            os.rename(
                os.path.join(config.savedir, 'persistent.bak'+str(i)),
                os.path.join(config.savedir, 'persistent.bak'+str(i+1)))
        except:
            pass
    try:
        os.rename(
            os.path.join(config.savedir, 'persistent.dat'),
            os.path.join(config.savedir, 'persistent.bak0'))
    except:
        pass

def persistent_write(self, filename):
    f = open(filename+'.new','wb')
    f.write('\n'.join(persistent_serialize(self)))
    f.close()
    try:
        os.unlink(filename)
    except:
        pass
    os.rename(filename+'.new',filename)

def persistent_save():
    if not renpy.game.persistent._changed:
        print "Persistent not changed, saving anyhow?"
        # If nothing's been changed, no need to save anything.
    persistent_cycle()
    try:
        persistent_write(renpy.game.persistent,os.path.join(os.path.expanduser(renpy.config.savedir),'persistent.dat'))
    except Exception,e:
        warnings.append((4,"Error saving persistent.dat",e))
        raise

def i64enc(i):
    if i<=0:
        if not i:
            return '0'
        raise Exception('i64 encoding only supports positive numbers.')
    ret = ''
    while i:
        ret = chr((i&0x3f)+0x30)+ret
        i >>= 6
    return ret

def i64dec(s):
    a = 0
    for c in s:
        a <<= 6
        a += ord(c)-0x30
    return a

class SanitizableRevertableObject(renpy.python.RevertableObject):
    # Subclasses must use a _member_sanitizers_ map; we don't specify one
    # here because we don't have any defaults that would make sense to inherit.

    # _member_serializers_ contains override functions to replace the default
    # serialization behaviour.
    _member_serializers_ = ()

    # _discardable_members_ lists members that should be thrown out when saved.
    _discardable_members_ = ()

    def Serialize(self,memoset,infix='',memoid=None, rollobjmemo=None):
        if memoid is None:
            memoid = id(self)
        t = type(self)
        if memoid in memoset:
            return ['sroref'+infix+' '+i64enc(memoid)]

        memoset.add(memoid)
        ret = ['sro'+infix+' '+t.__name__]
        ret.append('id '+i64enc(memoid))
        if memoid in rollobjmemo:
            self = self.RolledBackClone(rollobjmemo[memoid])
        for k in self._member_sanitizers_:
            if hasattr(self,k):
                if k in self._discardable_members_:
                    # Skip everything that we're supposed to discard.
                    continue
                v = getattr(self, k)
                if hasattr(t,k) and (v is getattr(t,k)):
                    # Skip values which are defaults.
                    continue
                elif k in self._member_serializers_:
                    ret.extend(self._member_serializers_[k](k,v,memoset,rollobjmemo=rollobjmemo))
                else:
                    ret.extend(serialize_tnv(k,v,memoset,rollobjmemo=rollobjmemo))
        if DEBUG:
            for k in self.__dict__:
                if (k not in self._discardable_members_) and (k not in self._member_sanitizers_):
                    serialize_warnings.append("Discarded field "+str(type(self))+"."+k)
                    ret.append("#skipped "+str(type(self))+"."+k)
        ret.append('# end sro '+(type(self).__name__))
        ret.append('end')
        return tuple(ret)

    if renpy.version_tuple<(6,99,12,2):
        def RolledBackClone(self, rollbackinfo):
            try:
                ret = type(self).__new__(type(self))
                ret.rollback(rollbackinfo)
            except Exception,e:
                print "Could not serialize "+str(type(self))+": "+str(e)
                raise
            return ret
    else:
        def RolledBackClone(self, rollbackinfo):
            try:
                ret = type(self).__new__(type(self))
                ret._rollback(rollbackinfo)
            except Exception,e:
                print "Could not serialize "+str(type(self))+": "+str(e)
                raise
            return ret

    def AfterLoad(self):
        # Null, just there so children can override it.
        pass
